/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <ctype.h> /* isprint() */

#include "cdw_debug.h"
#include "cdw_form.h"
#include "cdw_widgets.h"
#include "cdw_string.h"

static void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi);
static void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off);
static bool cdw_form_is_return_key(cdw_form_t *form, int key);

static bool cdw_form_is_checkbox_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_dropdown_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_button_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_input_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_safe_input_line_index(cdw_form_t *cdw_form, int fi);
/* To be removed.
static bool cdw_form_is_checkbox_index(cdw_form_t *cdw_form, int fi);
*/




static cdw_rv_t cdw_form_description_to_static_label(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);
static cdw_rv_t cdw_form_description_to_checkbox(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);
static cdw_rv_t cdw_form_description_to_dropdown(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);
static cdw_rv_t cdw_form_description_to_button(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);
static cdw_rv_t cdw_form_description_to_text(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);
static cdw_rv_t cdw_form_description_to_dynamic_label(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);




/**
   \brief Constructor

   \param n_fields - expected maximal number of fields in a form
*/
cdw_form_t *cdw_form_new(int n_fields)
{
	cdw_form_t *cdw_form = (cdw_form_t *) malloc(sizeof (cdw_form_t));
	if (!cdw_form) {
		cdw_vdm ("ERROR: failed to allocate memory for new cdw form\n");
		return (cdw_form_t *) NULL;
	}
	cdw_form->n_fields = n_fields;

	cdw_form->field_widget_types = malloc((size_t) cdw_form->n_fields * sizeof (cdw_id_t));
	if (!cdw_form->field_widget_types) {
		cdw_vdm ("ERROR: failed to allocate fields widget ids table\n");
		free(cdw_form);
		cdw_form = (cdw_form_t *) NULL;
		return (cdw_form_t *) NULL;
	}

	cdw_form->window = (WINDOW *) NULL;
	cdw_form->subwindow = (WINDOW *) NULL;
	cdw_form->form = (FORM *) NULL;
	cdw_form->fields = (FIELD **) NULL;

	cdw_form->form_id = 0;
	cdw_form->n_return_keys = 0;
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		cdw_form->return_keys[i] = 0;
	}

	return cdw_form;
}





/**
   \brief cdw form destructor - stage 2
*/
void cdw_form_delete(cdw_form_t **cdw_form)
{
	cdw_assert (cdw_form != (cdw_form_t **) NULL, "ERROR: incorrect argument\n");
	if (*cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}

	if ((*cdw_form)->field_widget_types != (cdw_id_t *) NULL) {
		free((*cdw_form)->field_widget_types);
		(*cdw_form)->field_widget_types = (cdw_id_t *) NULL;
	}

	free(*cdw_form);
	*cdw_form = (cdw_form_t *) NULL;

	return;
}





/**
   \brief cdw form destructor - stage 1

   Variable of type cdw_form_t needs to be deallocated
   in stages, this is first of them;
   second stage is cdw_form_delete()
*/
void cdw_form_delete_form_objects(cdw_form_t *cdw_form)
{
	if (!cdw_form) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}
	/* From ncurses docs: "The functions free_field() and
	   free_form are available to free field and form objects.
	   It is an error to attempt to free a field connected to
	   a form, but not vice-versa; thus, you will generally
	   free your form objects first." */
	if (cdw_form->form) {
		unpost_form(cdw_form->form);
		free_form(cdw_form->form);
		cdw_form->form = (FORM *) NULL;
	}

	for (int i = 0; i < cdw_form->n_fields; i++) {
		/* I forgot why I do this 'if (cdw_form->fields[i])'
		   test here.  Most probably because on failure to
		   build a cdw_form the fields[] table may be
		   sparse. */
		if (cdw_form->fields[i]) {
			if (cdw_form->field_widget_types[i] == CDW_WIDGET_ID_CHECKBOX) {
				CDW_CHECKBOX *cb = (CDW_CHECKBOX *) field_userptr(cdw_form->fields[i]);
				cdw_assert (cb, "ERROR: field #%d is checkbox, but the pointer is null\n", i);
				cdw_checkbox_free(cb);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_ID_BUTTON) {
				CDW_BUTTON *b = (CDW_BUTTON *) field_userptr(cdw_form->fields[i]);
				cdw_assert (b, "ERROR: field #%d is button, but the pointer is null\n", i);
				cdw_button_free(b);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_ID_DROPDOWN) {
				CDW_DROPDOWN *dd = (CDW_DROPDOWN *) field_userptr(cdw_form->fields[i]);
				cdw_assert (dd, "ERROR: field #%d is dropdown, but the pointer is null\n", i);
				cdw_dropdown_free(dd);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_ID_DYNAMIC_LABEL) {
				CDW_DYNAMIC_LABEL *label = (CDW_DYNAMIC_LABEL *) field_userptr(cdw_form->fields[i]);
				cdw_assert (label, "ERROR: field #%d is label, but the pointer is null\n", i);
				cdw_dynamic_label_delete(&label);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_ID_SAFE_INPUT_LINE) {
				CDW_SAFE_INPUT_LINE *input_line = (CDW_SAFE_INPUT_LINE *) field_userptr(cdw_form->fields[i]);
				cdw_assert (input_line, "ERROR: field #%d is input_line, but the pointer is null\n", i);
				cdw_safe_input_line_delete(&input_line);
			} else {
				;
			}
			free_field(cdw_form->fields[i]);
			cdw_form->fields[i] = (FIELD *) NULL;
		}
	}

	return;
}





/**
   \brief New style driver, for widgets that support ->driver() function

   This driver is simpler than cdw_form_driver_old(), mainly because it
   does not handle that many keys. It is only concerned with
   inter-field movement keys and most important keys: Enter, Escape
   and Exit.

   Ideally all forms used by cdw should migrate to this driver.

   In order for this function to work properly, all widgets in given
   \p cdw_form must support ->driver().

   \param cdw_form - cdw form to control
   \param initial_fi - index of widget that will initially have keyboard focus

   \return CDW_KEY_ENTER if one of widgets' driver has returned CDW_KEY_ENTER
   \return CDW_KEY_ESCAPE if one of widgets' driver has returned CDW_KEY_ESCAPE
   \return CDW_KEY_EXIT if one of widgets' driver has returned CDW_KEY_EXIT
*/
int cdw_form_driver(cdw_form_t *cdw_form, int initial_fi)
{
	cdw_form_driver_go_to_field(cdw_form, initial_fi);
	/* TODO: why this assignment? */
	initial_fi = field_index(current_field(cdw_form->form));
	cdw_form_driver_focus_on(cdw_form, initial_fi);


	int key = 'a'; /* Safe initial value. */
	int fi = initial_fi;

	while (key != CDW_KEY_ESCAPE && key != CDW_KEY_ENTER && key != KEY_EXIT) {

		/* Call a widget-specific driver, and don't care about
		   widget-specific behaviour. The widget-specific
		   behaviour is defined in widget module.

		   TODO: two widget drivers below accept "cdw_form_t
		   *" variable as second arg. Consider converting the
		   type of second arg to all widget's drivers from
		   "void *" to "cdw_form_t *".
		*/
		void *widget = field_userptr(cdw_form->fields[fi]);
		if (cdw_form_is_button_field_index(cdw_form, fi)) {
			CDW_BUTTON *b = (CDW_BUTTON *) widget;
			key = cdw_button_driver(b, (void *) cdw_form);

		} else if (cdw_form_is_dropdown_field_index(cdw_form, fi)) {
			CDW_DROPDOWN *dd = (CDW_DROPDOWN *) widget;
			key = cdw_dropdown_driver(dd, NULL);

		} else if (cdw_form_is_safe_input_line_index(cdw_form, fi)) {
			CDW_SAFE_INPUT_LINE *line = (CDW_SAFE_INPUT_LINE *) widget;
			key = cdw_safe_input_line_driver(line, NULL);

		} else if (cdw_form_is_checkbox_field_index(cdw_form, fi)) {
			CDW_CHECKBOX *cb = (CDW_CHECKBOX *) widget;
			key = cdw_checkbox_driver(cb, (void *) cdw_form);

		} else {
			/* May be that the only widget in the form is
			   a widget without a driver (e.g. TEXT
			   widget). What then? Here is a solution: */
			key = wgetch(cdw_form->subwindow);
		}



		/* Handle movement between widgets. */


		if (cdw_form_is_movement_key(key)) {
			/* We are moving away from a
			   widget. Un-highlight it to show that
			   keyboard focus is no longer on the
			   widget. */
			int old_fi = field_index(current_field(cdw_form->form));
			cdw_form_driver_focus_off(cdw_form, old_fi);



			/* Move to next or previous widget in the form. */
			if (key == KEY_DOWN || key == CDW_KEY_TAB) {
				form_driver(cdw_form->form, REQ_NEXT_FIELD);
				form_driver(cdw_form->form, REQ_END_LINE);

			} else if (key == KEY_UP || key == KEY_BTAB
				   /* || key == KEY_BACKSPACE */
				   ) {
				form_driver(cdw_form->form, REQ_PREV_FIELD);
				form_driver(cdw_form->form, REQ_END_LINE);

			} else {
				cdw_vdm ("ERROR: unhandled movement key %d/%s\n", key, cdw_ncurses_key_label(key));
			}



			/* We are moving into a next widget. Highlight it to
			   show that keyboard focus has moved to the
			   widget. */
			fi = field_index(current_field(cdw_form->form));
			cdw_form_driver_focus_on(cdw_form, fi);


		} else if (cdw_form_is_return_key(cdw_form, key)) {
			int ret_fi = field_index(current_field(cdw_form->form));
			if (isprint(key)
			    && cdw_form_is_input_field_index(cdw_form, ret_fi)) {
				/* Keys like 'q'/'Q' may be configured
				   as return keys, but they shouldn't
				   be considered as such in regular
				   text input fields. */
				;
			} else {
				break;
			}
		} else {
			;
		}
	}

	return key;
}





bool cdw_form_is_return_key(cdw_form_t *form, int key)
{
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		if (form->return_keys[i] == 0) {
			break;
		} else if (form->return_keys[i] == key) {
			return true;
		} else {
			;
		}
	}
	return false;
}





void cdw_form_add_return_char(cdw_form_t *form, int key)
{
	cdw_assert (form, "ERROR: \"form\" argument is NULL\n");
	cdw_assert (form->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    form->n_return_keys, N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key '0', but '0' is an initializer value\n");

	if (cdw_form_is_return_key(form, key)) {
		cdw_vdm ("WARNING: attempting to add key %d / \"%s\", but it is already on the list of return keys\n",
			 key, cdw_ncurses_key_label(key));
	} else {
		form->return_keys[form->n_return_keys++] = key;
	}

	return;
}





void cdw_form_add_return_chars(cdw_form_t *form, ...)
{
	cdw_assert (form, "ERROR: \"form\" argument is NULL\n");
	cdw_assert (form->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    form->n_return_keys, N_RETURN_KEYS_MAX);

	va_list ap;
	va_start(ap, form);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		cdw_form_add_return_char(form, key);
	}
	va_end(ap);

	return;
}




typedef cdw_rv_t (* cdw_form_converter_t)(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi);


/* Order of widget constructors/converters matches enumeration of
   widget types in cdw_widget.h. */
cdw_form_converter_t cdw_form_converters[] = {
	NULL,
	cdw_form_description_to_static_label,
	cdw_form_description_to_checkbox,
	cdw_form_description_to_dropdown,
	cdw_form_description_to_button,
	cdw_form_description_to_text,
	cdw_form_description_to_safe_input_line,
	cdw_form_description_to_dynamic_label
};





cdw_rv_t cdw_form_description_to_fields(cdw_form_descr_t descr[], cdw_form_t *cdw_form)
{
	/* "<=" - to set proper value of guard in fields table */
	for (int i = 0; i <= cdw_form->n_fields; i++) {
		cdw_form->fields[i] = (FIELD *) NULL;
	}

	cdw_assert (cdw_form->field_widget_types != (cdw_id_t *) NULL, "ERROR: table of widget ids is NULL\n");

	for (int fi = 0; fi < cdw_form->n_fields; fi++) {
		cdw_id_t widget_type = descr[fi].widget_type;
		cdw_assert (fi == descr[fi].field_enum, "ERROR: wrong order of items: field iterator (%d) != field enum (%d)\n", fi, descr[fi].field_enum);
		if (widget_type == -1) { /* guard in form description table */
			cdw_assert (0, "ERROR: reached descr guard, so loop condition is incorrect; fi = %d, n_fields = %d\n", fi, cdw_form->n_fields);
			break;
		}

		cdw_assert (cdw_widget_is_valid_widget_id(widget_type), "ERROR: invalid widget type %lld\n", widget_type);
		cdw_vdm ("INFO: creating field #%d: %s\n", fi, cdw_widgets_widget_type_label(widget_type));

		cdw_rv_t crv = cdw_form_converters[widget_type](cdw_form, &descr[fi], fi);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: converter failed for widget #%d with id %lld (%s)\n",
				 fi, widget_type, cdw_widgets_widget_type_label(widget_type));
			return CDW_ERROR;
		}
	}

	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_static_label(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_vdm ("INFO: initial string = \"%s\"\n", (char *) descr->data1);

	cdw_form->fields[fi] = cdw_ncurses_new_label_field(descr->n_lines,
							   descr->n_cols,
							   descr->begin_y,
							   descr->begin_x,
							   (char *) descr->data1);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_STATIC_LABEL;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_checkbox(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_form->fields[fi] = cdw_ncurses_new_checkbox_field(descr->begin_y, descr->begin_x, (char *) NULL);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	/* I don't know why, but sometimes moving into a checkbox
	   field erases content displayed in overlaying window;
	   turning off O_PUBLIC seems to fix this problem */
	field_opts_off(cdw_form->fields[fi], O_PUBLIC);

	CDW_CHECKBOX *cb = cdw_checkbox_new(cdw_form->subwindow, descr->begin_y, descr->begin_x, descr->data2);
	if (!cb) {
		cdw_vdm ("ERROR: failed to create checkbox widget for field #%d\n", fi);
		return CDW_ERROR;
	}

	int rv = set_field_userptr(cdw_form->fields[fi], (void *) cb);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: can't set userptr for field #%d: %d (%s) \n",
			 fi, rv, cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_CHECKBOX;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_dropdown(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_form->fields[fi] = cdw_ncurses_new_input_field(descr->n_lines,
							   descr->n_cols,
							   descr->begin_y,
							   descr->begin_x,
							   (char *) NULL,
							   0,
							   CDW_NCURSES_INPUT_NONE,
							   CDW_COLORS_INPUT);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}


	cdw_form_dropdown_maker_t *makers = (cdw_form_dropdown_maker_t *) descr->data1;
	int ind = (int) descr->data2;

	CDW_DROPDOWN *dd = makers[ind](cdw_form->subwindow, descr->begin_y, descr->begin_x,
				       descr->n_cols);
	if (!dd) {
		cdw_vdm ("ERROR: failed to create dropdown widget for field #%d\n", fi);
		return CDW_ERROR;
	}

	int rv = set_field_userptr(cdw_form->fields[fi], (void *) dd);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: can't set userptr for field #%d: %d (%s) \n",
			 fi, rv, cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_DROPDOWN;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_button(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_form->fields[fi] = cdw_ncurses_new_input_field(descr->n_lines,
							   descr->n_cols,
							   descr->begin_y,
							   descr->begin_x,
							   (char *) NULL,
							   0, //descr->data2,
							   CDW_NCURSES_INPUT_NONE,
							   CDW_COLORS_INPUT);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	/* To avoid situation when user presses some alphanumeric key
	   and the key is entered into field under a button. */
	field_opts_off(cdw_form->fields[fi], O_EDIT);

	CDW_BUTTON *b = cdw_button_new(cdw_form->subwindow, descr->begin_y, descr->begin_x, descr->data1, descr->data2);
	if (!b) {
		cdw_vdm ("ERROR: failed to create button widget for field #%d\n", fi);
		return CDW_ERROR;
	}

	int rv = set_field_userptr(cdw_form->fields[fi], (void *) b);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: can't set userptr for field #%d: %d (%s) \n",
			 fi, rv, cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_BUTTON;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_text(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_form_text_maker_t *makers = (cdw_form_text_maker_t *) descr->data1;
	int ind = (int) descr->data2;
	char *message = makers[ind]((int) descr->n_cols);

	cdw_form->fields[fi] = cdw_ncurses_new_label_field(descr->n_lines, descr->n_cols,
							   descr->begin_y, descr->begin_x,
							   message);
	free(message);
	message = (char *) NULL;

	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	field_opts_off(cdw_form->fields[fi], O_ACTIVE);
	field_opts_off(cdw_form->fields[fi], O_EDIT);

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_TEXT;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_safe_input_line(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_vdm ("INFO: initial string = \"%s\"\n", (char *) descr->data1);
	cdw_form->fields[fi] = cdw_ncurses_new_input_field(descr->n_lines,
							   descr->n_cols,
							   descr->begin_y,
							   descr->begin_x,
							   (char *) descr->data1,
							   descr->data2,
							   CDW_NCURSES_INPUT_NONE,
							   CDW_COLORS_INPUT);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	CDW_SAFE_INPUT_LINE *line = cdw_safe_input_line_new(cdw_form->subwindow,
							    &(cdw_form->form),
							    cdw_form->fields[fi],
							    descr->n_cols,
							    (char *) descr->data1,
							    CDW_NCURSES_INPUT_NONE,
							    (int) descr->data2);
	if (!line) {
		cdw_vdm ("ERROR: failed to create safe input line widget for field #%d\n", fi);
		return CDW_ERROR;
	}

	int rv = set_field_userptr(cdw_form->fields[fi], (void *) line);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: can't set userptr for field #%d: %d (%s) \n",
			 fi, rv, cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_SAFE_INPUT_LINE;
	return CDW_OK;
}





cdw_rv_t cdw_form_description_to_dynamic_label(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi)
{
	cdw_vdm ("INFO: initial string = \"%s\"\n", (char *) descr->data1);
	cdw_form->fields[fi] = cdw_ncurses_new_label_field(descr->n_lines, descr->n_cols,
							   descr->begin_y, descr->begin_x,
							   (char *) descr->data1);
	if (!cdw_form->fields[fi]) {
		cdw_vdm ("ERROR: failed to initialize field #%d\n", fi);
		return CDW_ERROR;
	}

	CDW_DYNAMIC_LABEL *label = cdw_dynamic_label_new(cdw_form->subwindow,
							 cdw_form->fields[fi],
							 descr->n_cols,
							 (char *) descr->data1);
	if (!label) {
		cdw_vdm ("ERROR: failed to create dynamic label widget for field #%d\n", fi);
		return CDW_ERROR;
	}

	int rv = set_field_userptr(cdw_form->fields[fi], (void *) label);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: can't set userptr for field #%d: %d (%s) \n",
			 fi, rv, cdw_ncurses_error_string(rv));
		return CDW_ERROR;
	}

	cdw_form->field_widget_types[fi] = CDW_WIDGET_ID_DYNAMIC_LABEL;
	return CDW_OK;
}





bool cdw_form_is_movement_key(int key)
{
	if (key == KEY_DOWN
	    || key == CDW_KEY_TAB
	    || key == KEY_UP
	    || key == KEY_BTAB

	    /* in some configurations backspace may be a movement key,
	       i.e. when cursor is at the beginning of input field */
	    /* || key == KEY_BACKSPACE */
	    ) {

		return true;
	} else {
		return false;
	}
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi)
{
	cdw_form_driver_toggle_focus(cdw_form, fi, true);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_on(cdw_form_t *cdw_form, int fi)
{
	cdw_form_driver_toggle_focus(cdw_form, fi, false);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	if (cdw_form_is_button_field_index(cdw_form, fi)) {
		CDW_BUTTON *button = cdw_form_get_button(cdw_form, fi);
		if (button) {
			void (*function)(CDW_BUTTON *) = (void (*)(CDW_BUTTON *)) NULL;
			if (off) {
				function = cdw_button_unfocus;
				/* moving away from button, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_button_focus;
				/* moving into button, don't display blinking cursor */
				curs_set(0);
			}
			function(button);
		}
	} else if (cdw_form_is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form_get_dropdown(cdw_form, fi);
		if (dropdown) {
			void (*function)(CDW_DROPDOWN *) = (void (*)(CDW_DROPDOWN *)) NULL;
			if (off) {
				function = cdw_dropdown_unfocus;
				/* moving away from dropdown, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_dropdown_focus;
				/* moving into dropdown, don't display blinking cursor */
				curs_set(0);
			}
			function(dropdown);
		}
	} else if (cdw_form_is_checkbox_field_index(cdw_form, fi)
		   || cdw_form_is_input_field_index(cdw_form, fi)) {
		if (off) {
			/* moving away from checkbox/input, don't display blinking cursor */
			curs_set(0);
		} else {
			/* moving away from checkbox/input, display blinking cursor */
			curs_set(1);
		}
	} else {
		;
	}

	//form_driver(cdw_form->form, REQ_INS_MODE);
	wrefresh(cdw_form->subwindow);

	return;
}





CDW_DROPDOWN *cdw_form_get_dropdown(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_DROPDOWN,
		    "ERROR: field index %d: widget type is not DROPDOWN, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	CDW_DROPDOWN *dd = (CDW_DROPDOWN *) field_userptr(cdw_form->fields[fi]);
	cdw_assert (dd, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_assert (dd->widget_id == CDW_WIDGET_ID_DROPDOWN,
		    "ERROR: field index %d: internal widget id is not DROPDOWN, is %s\n",
		    fi, cdw_widgets_widget_type_label(dd->widget_id));

	return dd;
}





CDW_DYNAMIC_LABEL *cdw_form_get_dynamic_label(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_DYNAMIC_LABEL,
		    "ERROR: field index %d: widget type is not DYNAMIC_LABEL, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	CDW_DYNAMIC_LABEL *label = (CDW_DYNAMIC_LABEL *) field_userptr(cdw_form->fields[fi]);
	cdw_assert (label, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_assert (label->widget_id == CDW_WIDGET_ID_DYNAMIC_LABEL,
		    "ERROR: field index %d: internal widget id is not DYNAMIC_LABEL, is %s\n",
		    fi, cdw_widgets_widget_type_label(label->widget_id));

	return label;
}





CDW_SAFE_INPUT_LINE *cdw_form_get_safe_input_line(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_SAFE_INPUT_LINE,
		    "ERROR: field index %d: widget type is not SAFE_INPUT_LINE, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	CDW_SAFE_INPUT_LINE *line = (CDW_SAFE_INPUT_LINE *) field_userptr(cdw_form->fields[fi]);
	cdw_assert (line, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_assert (line->widget_id == CDW_WIDGET_ID_SAFE_INPUT_LINE,
		    "ERROR: field index %d: internal widget id is not SAFE_INPUT_LINE, is %s\n",
		    fi, cdw_widgets_widget_type_label(line->widget_id));

	return line;
}





/**
   \brief Get string from an input line widget

   The function works for both older INPUT_LINE widget, as well as for
   newer "SAFE_INPUT_LINE" widget.

   String is trimmed from right side. Returned pointer is owned by
   this function (by cdw_form, to be more precise). Do not free() it.

   \param cdw_form - form in which an input line widget is embedded
   \param fi - field index, index of the widget in the form

   \return pointer to string stored in input line widget
*/
const char *cdw_form_get_string(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_SAFE_INPUT_LINE,

		    "ERROR: field index %d: widget type is not input, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	char *string = field_buffer(cdw_form->fields[fi], 0);
	cdw_assert (string, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_string_rtrim(string);

	return string;
}





CDW_BUTTON *cdw_form_get_button(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_BUTTON,
		    "ERROR: field index %d: widget type is not BUTTON, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	CDW_BUTTON *button = (CDW_BUTTON *) field_userptr(cdw_form->fields[fi]);
	cdw_assert (button, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_assert (button->widget_id == CDW_WIDGET_ID_BUTTON,
		    "ERROR: field index %d: internal widget id is not BUTTON, is %s\n",
		    fi, cdw_widgets_widget_type_label(button->widget_id));

	return button;
}





CDW_CHECKBOX *cdw_form_get_checkbox(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_CHECKBOX,
		    "ERROR: field index %d: widget type is not CHECKBOX, is %s\n",
		    fi, cdw_widgets_widget_type_label(cdw_form->field_widget_types[fi]));

	cdw_assert (cdw_form->fields[fi], "ERROR: field %d is NULL\n", fi);


	CDW_CHECKBOX *cb = (CDW_CHECKBOX *) field_userptr(cdw_form->fields[fi]);
	cdw_assert (cb, "ERROR: field index %d: field user pointer is NULL\n", fi);
	cdw_assert (cb->widget_id == CDW_WIDGET_ID_CHECKBOX,
		    "ERROR: field index %d: internal widget id is not CHECKBOX, is %s\n",
		    fi, cdw_widgets_widget_type_label(cb->widget_id));

	return cb;
}





bool cdw_form_is_checkbox_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_CHECKBOX;
}





bool cdw_form_is_dropdown_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_DROPDOWN;
}





bool cdw_form_is_safe_input_line_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_SAFE_INPUT_LINE;
}




#if 0 /* To be removed. */
bool cdw_form_is_checkbox_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_CHECKBOX;
}
#endif




bool cdw_form_is_button_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_BUTTON;
}





bool cdw_form_is_input_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi >= 0 && fi < cdw_form->n_fields,
		    "ERROR: field index %d is out of range (n_fields = %d)\n",
		    fi, cdw_form->n_fields);

	return cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_SAFE_INPUT_LINE;
}





void cdw_form_driver_go_to_field(cdw_form_t *cdw_form, int fi)
{
	FORM *form = cdw_form->form;
	if (fi < 0 || fi >= cdw_form->n_fields) {
		cdw_vdm ("WARNING: fi = %d out of bounds (n_fields = %d)\n", fi, cdw_form->n_fields);
		fi = 0;
	}

	int rv = form_driver(form, REQ_FIRST_FIELD);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
	}

	/* initial field may not be first when driver needs to switch
	   to field in which there is an error */
	/* TODO: check why using simple for() loop fails: iterating stops
	   before correct field */
	int i = field_index(current_field(form));
	if (i == fi) {
		;
	} else {
		while (i < fi && i < cdw_form->n_fields - 1) {
			rv = form_driver(form, REQ_NEXT_FIELD);
			if (rv != E_OK) {
				cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
				break;
			} else {
				i = field_index(current_field(form));
			}
		}
	}

	cdw_form_driver_focus_on(cdw_form, fi);

	return;
}





void cdw_form_redraw_widgets(cdw_form_t *cdw_form)
{
	for (int i = 0; i < cdw_form->n_fields; i++) {
		cdw_id_t id = cdw_form->field_widget_types[i];
		if (id == CDW_WIDGET_ID_CHECKBOX) {
			CDW_CHECKBOX *cb = cdw_form_get_checkbox(cdw_form, i);
			cdw_checkbox_draw(cb);
		} else if (id == CDW_WIDGET_ID_BUTTON) {
			CDW_BUTTON *b = cdw_form_get_button(cdw_form, i);
			/* TODO: what if the button has keyboard
			   focus? With current code it would appear as
			   not having the focus. */
			cdw_button_unfocus(b);
		} else if (id == CDW_WIDGET_ID_DROPDOWN) {
			CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, i);
			cdw_dropdown_display_current_item(dd);
		} else {
			;
		}

	}
	return;
}





bool cdw_form_get_checkbox_state(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_ID_CHECKBOX,
		    "ERROR: trying to get state of field which isn't a CHECKBOX\n");

	CDW_CHECKBOX *cb = cdw_form_get_checkbox(cdw_form, fi);
	return cdw_checkbox_get_state(cb);
}





/**
   \brief Set a callback function for a widget

   For different widgets the callback is called on different actions.

   In case of button widgets, use this function only to set some
   interesting callback. A button's callback that only returns a key
   pressed while keyboard focus was on the button (e.g. Escape or
   Enter key) is not an interesting callback. To configure key
   returned by button on such occasion, use
   cdw_form_set_button_return_key().

   An interesting callback function opens options window, reads
   contents of file, displays file picker dialog. Use this function to
   configure such an interesting callback for a widget.
*/
void *cdw_form_set_widget_callback(cdw_form_t *cdw_form, int fi, cdw_form_widget_function_t function)
{
	cdw_id_t id = cdw_form->field_widget_types[fi];
	if (id == CDW_WIDGET_ID_BUTTON) {
		CDW_BUTTON *b = field_userptr(cdw_form->fields[fi]);
		b->on_click_callback = function;
		set_field_userptr(cdw_form->fields[fi], (void *) b);
		return (void *) b;
	} else if (id == CDW_WIDGET_ID_CHECKBOX) {
		CDW_CHECKBOX *cb = field_userptr(cdw_form->fields[fi]);
		cb->on_toggle_callback = function;
		set_field_userptr(cdw_form->fields[fi], (void *) cb);
		return (void *) cb;
	} else if (id == CDW_WIDGET_ID_DROPDOWN) {
		CDW_DROPDOWN *dd = field_userptr(cdw_form->fields[fi]);
		dd->on_select_callback = function;
		set_field_userptr(cdw_form->fields[fi], (void *) dd);
		return (void *) dd;
	} else {
		cdw_assert (0, "ERROR: called the function for widget that doesn't have a callback\n");
		return (void *) NULL;
	}
}





/**
   \brief Bind instances of two specific widgets together

   Safe input line widget requires a reference to dynamic label widget
   in order to work properly (to display error message on insecure
   character). This function makes the connection.

   The two widgets are embedded in a cdw form, so they are identified
   by indexes referring to the form.

   \param cdw_form - form in which the two widgets are embedded
   \param safe_input_fi index of form field associated with CDW_WIDGET_ID_SAFE_INPUT_LINE widget
   \param dynamic_label_fi index of form field associated with CDW_WIDGET_ID_DYNAMIC_LABEL widget
*/
void cdw_form_bind_input_and_label(cdw_form_t *cdw_form, int safe_input_fi, int dynamic_label_fi)
{
#if 0
	cdw_id_t i_id = cdw_form->field_widget_types[safe_input_fi];
	cdw_assert (i_id == CDW_WIDGET_ID_SAFE_INPUT_LINE, "ERROR: the widget is not safe input line\n");
	cdw_id_t l_id = cdw_form->field_widget_types[dynamic_label_fi];
	cdw_assert (l_id == CDW_WIDGET_ID_DYNAMIC_LABEL, "ERROR: the widget is not dynamic label\n");
#endif

	CDW_SAFE_INPUT_LINE *line = cdw_form_get_safe_input_line(cdw_form, safe_input_fi);
	CDW_DYNAMIC_LABEL *label = cdw_form_get_dynamic_label(cdw_form, dynamic_label_fi);

	cdw_safe_input_line_bind_message_area(line, label);

	set_field_userptr(cdw_form->fields[safe_input_fi], (void *) line);

	return;
}





void *cdw_form_set_button_return_key(cdw_form_t *cdw_form, int fi, int return_key)
{
	cdw_id_t id = cdw_form->field_widget_types[fi];
	cdw_assert (id == CDW_WIDGET_ID_BUTTON, "ERROR: trying to add return key for widget other than button (id = %lld / %s).\n", id, cdw_widgets_widget_type_label(id));

	CDW_BUTTON *b = field_userptr(cdw_form->fields[fi]);
	b->on_click_key = return_key;
	set_field_userptr(cdw_form->fields[fi], (void *) b);
	return (void *) b;
}





/**
   \brief Replace a field in existing cdw_form with new field

   To be more precise, old widget is replaced with new one. The
   ncurses FIELD field is being recreated in the process as well.

   Notice that the cdw_form is not recreated from scratch, just a part
   of it.

   In theory the function can replace old widget of any type with new
   widget of any other type, but for now the function only works (and
   has been tested) for old and new widget of type
   CDW_WIDGET_ID_SAFE_INPUT_LINE.
*/
cdw_rv_t cdw_form_replace_field(cdw_form_t *cdw_form, cdw_form_descr_t *descr, int fi, cdw_id_t old_widget_id, cdw_id_t new_widget_id)
{
	cdw_assert (old_widget_id == CDW_WIDGET_ID_SAFE_INPUT_LINE
		    && old_widget_id == CDW_WIDGET_ID_SAFE_INPUT_LINE,
		    "ERROR: for now only CDW_WIDGET_ID_SAFE_INPUT_LINE is supported\n");
	/* TODO: add rest of constructors. */

	cdw_assert (old_widget_id == cdw_form->field_widget_types[fi], "ERROR: old widget IDs don't match\n");


	/* Delete old widget. */
	if (old_widget_id == CDW_WIDGET_ID_SAFE_INPUT_LINE) {
		CDW_SAFE_INPUT_LINE *line = cdw_form_get_safe_input_line(cdw_form, fi);
		cdw_safe_input_line_delete(&line);
	}


	/* New ncurses FIELD field will be created in constructor
	   function called after this free_field(). */
	/* From ncurses docs: "The functions free_field() and
	   free_form are available to free field and form objects.  It
	   is an error to attempt to free a field connected to a form,
	   but not vice-versa; thus, you will generally free your form
	   objects first.".

	   So first unpost form, then free form, then free field. */
	int rv = unpost_form(cdw_form->form);
	cdw_vdm_n ("unpost_form", rv, "");
	rv = free_form(cdw_form->form);
	cdw_vdm_n ("free_form", rv, "");
	rv = free_field(cdw_form->fields[fi]);
	cdw_vdm_n ("free_field", rv, "");


	/* Create and insert new widget. */
	if (new_widget_id == CDW_WIDGET_ID_SAFE_INPUT_LINE) {
		cdw_form_description_to_safe_input_line(cdw_form, descr, fi);
	}

	cdw_form->form = cdw_ncurses_new_form(cdw_form->window,
					      cdw_form->subwindow,
					      cdw_form->fields);
	/* Make new widget become visible and usable in cdw form. */
	rv = set_form_fields(cdw_form->form, cdw_form->fields);
	cdw_vdm ("INFO: set_form_fields() returns %s\n", cdw_ncurses_error_string(rv));
	post_form(cdw_form->form);

	return CDW_OK;
}
