/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_SETTINGS_H
#define MEDIASCANNER_SETTINGS_H

// C++ Standard Library
#include <map>
#include <string>
#include <utility>
#include <vector>
#include <functional>

// Media Scanner Library
#include "mediascanner/glibutils.h"

namespace mediascanner {

class Settings {
public:
    class KeyName {
        friend class Settings;

    protected:
        explicit KeyName(const std::string &name)
            : name_(name) {
        }

    private:
        const std::string name_;
    };

    template<typename T> class Key : public KeyName {
        friend class Settings;

    private:
        typedef T value_type;

        explicit Key(const std::string &name)
            : KeyName(name) {
        }
    };

    struct MediaFormat {
        MediaFormat() {
        }

        MediaFormat(const std::string &name,
                    const std::string &caps)
            : name(name)
            , caps(caps) {
        }

        std::string name;
        std::string caps;
    };

    struct MetadataSource {
        typedef std::map<std::string, std::string> Config;

        MetadataSource() {
        }

        MetadataSource(const std::string &plugin_id,
                       const std::string &source_id,
                       const Config &config)
            : plugin_id(plugin_id)
            , source_id(source_id)
            , config(config) {
        }

        std::string plugin_id;
        std::string source_id;
        Config config;
    };

    typedef std::function<void()> ChangeListener;
    typedef std::vector<MediaFormat> MediaFormatList;
    typedef std::vector<MetadataSource> MetadataSourceList;
    typedef std::vector<std::string> StringList;

    static const Key<MediaFormatList> kMandatoryContainers;
    static const Key<MediaFormatList> kMandatoryDecoders;
    static const Key<MetadataSourceList> kMetadataSources;
    static const Key<StringList> kMediaRoots;

    Settings();

    MediaFormatList mandatory_containers() const;
    MediaFormatList mandatory_decoders() const;
    MetadataSourceList metadata_sources() const;
    StringList media_root_urls() const;
    StringList media_root_paths() const;

    template<typename T>
    T lookup(const Key<T> &key) const;

    unsigned connect(const KeyName &key, const ChangeListener &listener) const;
    void disconnect(unsigned handler_id) const;

    std::vector<std::string> LoadMetadataSources() const;
    static std::vector<std::string> LoadMetadataSources
                                        (const MetadataSourceList &sources);

private:
    Wrapper<GSettings> settings_;
};

} // namespace mediascanner

#endif // MEDIASCANNER_SETTINGS_H
