/*
  Copyright (C) 2008 André Gaul, Jan Friederich, Steffen Basting, Kai Hertel

        This file is part of mmpong.

        mmpong is free software: you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version.

        mmpong is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <SDL_opengl.h>
#include <stdexcept>
#include <iostream>
#include "resource.h"
#include "renderer.h"
#include "glhelper.h"


#define BALL_RADIUS 0.03f
#define PADDLE_RADIUS 0.05f
#define BOUNDARY_RADIUS 0.05f
#define BLUR_DECAY 0.15f
#define DIGIT_WIDTH (40.f/512.f)
#define DIGIT_RATIO (40.f/64.f)
#define SCORE_WIDTH (192.f/512.f)
#define SCORE_RATIO (192.f/64.f)
#define SCORE_XOFFSET (0.0f)
#define SCORE_FONT_SIZE 0.07f
#define PLAYERS_WIDTH (264.f/512.f)
#define PLAYERS_RATIO (264.f/64.f)
#define PLAYERS_XOFFSET (192.f/512.f)
#define PLAYERS_FONT_SIZE (0.04f)

using namespace std;

Renderer::Renderer(int _win_w, int _win_h, bool _full, bool _sexy, const string &respath, const string &title){
	win_w = _win_w;
	win_h = _win_h;
	sexy = _sexy;
	full = _full;

	if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0) 
		throw runtime_error(string("Renderer: Unable to initialize SDL: ") + SDL_GetError());

	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);
	//SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE, 16);
	
	if (sexy) {
		SDL_GL_SetAttribute(SDL_GL_ACCUM_RED_SIZE, 4);
		SDL_GL_SetAttribute(SDL_GL_ACCUM_GREEN_SIZE, 4);
		SDL_GL_SetAttribute(SDL_GL_ACCUM_BLUE_SIZE, 4);
		SDL_GL_SetAttribute(SDL_GL_ACCUM_ALPHA_SIZE, 4);
	}

	// use SDL_GetVideoInfo call before SDL_SetVideoMode to get screen resolution
	const SDL_VideoInfo *info = SDL_GetVideoInfo();
	if (!info)
		throw runtime_error("Renderer: Unable to get video info");
	full_w = info->current_w;
	full_h = info->current_h;
	video_flags = SDL_OPENGL | (info->hw_available ? SDL_HWSURFACE : SDL_SWSURFACE);
	
	SDL_WM_SetCaption(title.c_str(), NULL);

	SDL_Surface *icon = IMG_Load(RESOURCE("icon/mmpong.png"));
	if (icon)
		SDL_WM_SetIcon(icon, NULL);
	else
		cerr << "could not load bitmap: "<< RESOURCE("icon/mmpong.png") << endl;

	SDL_ShowCursor(0);
	
	resize();
# if !(defined(__APPLE__) || defined(WIN32))
		init_graphics();
# endif
}


Renderer::~Renderer() {
	glDeleteTextures(1, &score_texture);
	glDeleteTextures(1, &field_texture);
	glDeleteLists(box_list, 1);
	glDeleteLists(cylinder_list, 1);
	glDeleteLists(ball_list, 1);

	SDL_Quit();
}


void Renderer::resize() {
	int width=get_width(), height=get_height();
	if ((surface = SDL_SetVideoMode(width, height, 0, video_flags | (full ? SDL_FULLSCREEN : SDL_RESIZABLE))) == NULL) 
		throw runtime_error(std::string("Renderer: Unable to create OpenGL screen: ") + SDL_GetError());

	/* This is a hack for Windows and SDL 1.2:
	   In Windows, the OpenGL state seems to get lost every time we
	   call SDL_SetVideoMode(). That is, we have to reset the state
	   which unfortunately include loading the textures from files etc.
	   Hopefully this will be fixed in SDL 1.3.
	   Note that calling SDL_SetVideoMode() is necessary for the graphics
	   to be displayed properly and the mouse events to provide correct
	   data.
	 */

#if (defined(__APPLE__) || defined(WIN32))
	init_graphics();
#endif

	if (width < height) {
		glViewport(0, (height-width)/2, width, width);
		height = width;
	} else
		glViewport(0, 0, width, height);

	if (height <= 0) height = 1;
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	// z-planes need some tuning
	gluPerspective(45.0f, ((float)width)/height, 0.001f, 10.0f);
	glMatrixMode(GL_MODELVIEW);

	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_ACCUM_BUFFER_BIT);

	field_width = 0.9f*((float)width)/height;
}


void Renderer::init_graphics() {
	glShadeModel(GL_SMOOTH);
	glEnable(GL_LIGHTING);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	//glEnable(GL_NORMALIZE);
	//glEnable(GL_DEPTH_TEST);
	//glDepthFunc(GL_LEQUAL);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	GLfloat ambient[] = {0.2f, 0.2f, 0.2f, 0.5f};
	GLfloat diffuse[] = {0.8f, 0.8f, 0.8f, 1.0f};
	GLfloat position[] = {0.0f, 1.0f, 1.0f, 0.0f};
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position);
	glEnable(GL_LIGHT1);

	create_box(&box_list);
	create_cylinder(&cylinder_list, 16);
	create_ball(&ball_list, 4, 16);

	if (load_png(&score_texture, RESOURCE("images/score.png")) != 0)
		cerr << "Renderer: Unable to load texture: " << RESOURCE("images/score.png") << endl;
	if (load_png(&field_texture, RESOURCE("images/field.png")) != 0)
		cerr << "Renderer: Unable to load texture: " << RESOURCE("images/field.png") << endl;
}


void Renderer::render(Game &game, float pos) {
	GLfloat color[] = {0.0f, 0.0f, 0.0f, 0.0f};

	glClear(GL_COLOR_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.5f*field_width, 0.5f, 1.5f, 0.5f*field_width, 0.5f, 0.0f, 0.0f, 1.0f, 0.0f);

	glPushMatrix();
		glRotatef(-22.5f, 1.0f, 0.0f, 0.0f);

		if (sexy) {
			glAccum(GL_MULT, 1.0f-BLUR_DECAY);

			color[0] = 0.7f, color[1] = 0.7f, color[2] = 0.0f, color[3] = 1.0f;
			glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
			glPushMatrix();
				glTranslatef(game.get_game()->ball.pos[0]*field_width, game.get_game()->ball.pos[1], 0.0f);
				glScalef(2.0f*BALL_RADIUS, 2.0f*BALL_RADIUS, 2.0f*BALL_RADIUS);
				glCallList(ball_list);
			glPopMatrix();

			glAccum(GL_ACCUM, 1.0f);
			glAccum(GL_RETURN, 1.0f);
		}

		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, field_texture);
		color[0] = 0.1f, color[1] = 0.1f, color[2] = 0.1f, color[3] = 0.3f;
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		glPushMatrix();
			glScalef(field_width, 1.0f, 1.0f);
			glTranslatef(0.5f, 0.5f, 0.0f);
			glRotatef(45.0f, 0.0f, 0.0f, 1.0f);
			glBegin(GL_QUADS);
				glNormal3f(0.0f, 0.0f, 1.0f);
				glTexCoord2f(0.0f, 0.0f); glVertex3f(-0.6f, -0.2f, 0.0f);  // bottom left
				glTexCoord2f(1.0f, 0.0f); glVertex3f(0.6f, -0.2f, 0.0f);  // bottom right
				glTexCoord2f(1.0f, 1.0f); glVertex3f(0.6f, 0.2f, 0.0f);  // top right
				glTexCoord2f(0.0f, 1.0f); glVertex3f(-0.6f, 0.2f, 0.0f);  // top left
			glEnd();
		glPopMatrix();
		glDisable(GL_TEXTURE_2D);

		color[0] = 0.4f, color[1] = 0.4f, color[2] = 0.4f, color[3] = 0.2f;
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		glPushMatrix();
			glTranslatef(0.5f*field_width, 1.0f+0.5f*BOUNDARY_RADIUS+BALL_RADIUS, 0.0f);
			glScalef(field_width+2.0f*BALL_RADIUS+2.0f*PADDLE_RADIUS, BOUNDARY_RADIUS, BOUNDARY_RADIUS);
			glCallList(box_list);
		glPopMatrix();
		glPushMatrix();
			glTranslatef(0.5f*field_width, -0.5f*BOUNDARY_RADIUS-BALL_RADIUS, 0.0f);
			glScalef(field_width+2.0f*BALL_RADIUS+2.0f*PADDLE_RADIUS, BOUNDARY_RADIUS, BOUNDARY_RADIUS);
			glCallList(box_list);
		glPopMatrix();

		draw_paddle(pos, game.get_game()->pad[game.get_team()].size, game.get_team(), 1, game.get_game()->pad_attr[game.get_team()].profile);
		for (unsigned int i = 0; i < 2; ++i) {
			draw_paddle(game.get_game()->pad[i].mean, game.get_game()->pad[i].size, i, 0, game.get_game()->pad_attr[i].profile);
			draw_variance(game.get_game()->pad[i].mean, game.get_game()->pad[i].var, i);
		}

		// draw opaque ball in any case
		color[0] = 0.7f, color[1] = 0.7f, color[2] = 0.0f, color[3] = 1.0f;
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		//color[0] = 1.0f, color[1] = 1.0f, color[2] = 1.0f, color[3] = 1.0f;
		//glMaterialfv(GL_FRONT, GL_SPECULAR, color);
		//glMaterialf(GL_FRONT, GL_SHININESS, 30.0f);
		glPushMatrix();
			glTranslatef(game.get_game()->ball.pos[0]*field_width, game.get_game()->ball.pos[1], 0.0f);
			glScalef(2.0f*BALL_RADIUS, 2.0f*BALL_RADIUS, 2.0f*BALL_RADIUS);
			glCallList(ball_list);
		glPopMatrix();

	glPopMatrix();

	for (unsigned int i = 0; i < 2; ++i) {
		draw_score(i, game.get_game()->pad_attr[i].score);
		draw_players(i, game.get_game()->pad_attr[i].peers);
	}
}


void Renderer::draw_paddle(float mean, float size, uint16_t team, uint16_t mode, uint16_t profile) {
	GLfloat color[] = {team ? 1.0f : 0.0f, 0.0f, team ? 0.0f : 1.0, mode ? 1.0f : 0.5f};
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);

	glPushMatrix();
		/*if (mode) {
			glTranslatef(team ? field_width+0.5f*PADDLE_RADIUS+BALL_RADIUS : -0.5f*PADDLE_RADIUS-BALL_RADIUS, mean, 0.0f);
			glScalef(PADDLE_RADIUS, size, PADDLE_RADIUS);
		}
		else {
			glTranslatef(team ? field_width+0.25f*PADDLE_RADIUS+BALL_RADIUS : -0.25f*PADDLE_RADIUS-BALL_RADIUS, mean, 0.0f);
			glScalef(0.5f*PADDLE_RADIUS, size, PADDLE_RADIUS);
		}*/
		glTranslatef(team ? field_width+0.5f*PADDLE_RADIUS+BALL_RADIUS : -0.5f*PADDLE_RADIUS-BALL_RADIUS, mean, 0.0f);
		glScalef(PADDLE_RADIUS, size, PADDLE_RADIUS);

		if (profile) 
			glCallList(cylinder_list);
		else 
			glCallList(box_list);
	glPopMatrix();
};


void Renderer::draw_players(uint16_t team, uint16_t number) {
	GLfloat color[4] = {1.0f, 1.0f, 1.0f, 1.0f};

	//glPushAttrib(GL_TEXTURE_BIT);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, score_texture);

	//glEnable(GL_BLEND);
	//glDepthMask(GL_FALSE);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	//glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	if (!team) {
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		glPushMatrix();
			glTranslatef(0.5f*field_width, 1.04f, 0.0f);
			glBegin(GL_QUADS);
				glNormal3f(0.0f, 0.0f, 1.0f);
				glTexCoord2f(PLAYERS_XOFFSET, 0.0f); glVertex3f(-0.5f*PLAYERS_RATIO*PLAYERS_FONT_SIZE, 0.0f, 0.0f);  // bottom left
				glTexCoord2f(PLAYERS_XOFFSET+PLAYERS_WIDTH, 0.0f); glVertex3f(0.5f*PLAYERS_RATIO*PLAYERS_FONT_SIZE, 0.0f, 0.0f);  // bottom right
				glTexCoord2f(PLAYERS_XOFFSET+PLAYERS_WIDTH, 0.5f); glVertex3f(0.5f*PLAYERS_RATIO*PLAYERS_FONT_SIZE, PLAYERS_FONT_SIZE, 0.0f);  // top right
				glTexCoord2f(PLAYERS_XOFFSET, 0.5f); glVertex3f(-0.5f*PLAYERS_RATIO*PLAYERS_FONT_SIZE, PLAYERS_FONT_SIZE, 0.0f);  // top left
			glEnd();
		glPopMatrix();
	}

	uint16_t n = 1, m = number/10;
	while (m > 0) ++n, m/=10;

	color[0] = team ? 1.0f : 0.0f, color[1] = 0.0f, color[2] = team ? 0.0f : 1.0f;
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);

	glPushMatrix();
		glTranslatef(team ? 0.8f*field_width : 0.2f*field_width, 1.04f, 0.0f);
		glTranslatef(team ? -1.f*DIGIT_RATIO*PLAYERS_FONT_SIZE : (n-1)*DIGIT_RATIO*PLAYERS_FONT_SIZE, 0.0f, 0.0f);
		draw_digit(number%10, PLAYERS_FONT_SIZE);
		number /= 10;
		while (number > 0) {
			glTranslatef(-DIGIT_RATIO*PLAYERS_FONT_SIZE, 0.0f, 0.0f);
			draw_digit(number%10, PLAYERS_FONT_SIZE);
			number /= 10;
		}
	glPopMatrix();

	glDisable(GL_TEXTURE_2D);
	//glPopAttrib(GL_TEXTURE_BIT);
};


void Renderer::draw_score(uint16_t team, uint16_t score) {
	GLfloat color[4] = {1.0f, 1.0f, 1.0f, 1.0f};

	//glPushAttrib(GL_TEXTURE_BIT);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, score_texture);

	//glEnable(GL_BLEND);
	//glDepthMask(GL_FALSE);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	//glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	if (!team) {
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		glPushMatrix();
			glTranslatef(0.5f*field_width, 0.94f, 0.0f);
			glBegin(GL_QUADS);
				glNormal3f(0.0f, 0.0f, 1.0f);
				glTexCoord2f(SCORE_XOFFSET, 0.0f); glVertex3f(-0.5f*SCORE_RATIO*SCORE_FONT_SIZE, 0.0f, 0.0f);  // bottom left
				glTexCoord2f(SCORE_XOFFSET+SCORE_WIDTH, 0.0f); glVertex3f(0.5f*SCORE_RATIO*SCORE_FONT_SIZE, 0.0f, 0.0f);  // bottom right
				glTexCoord2f(SCORE_XOFFSET+SCORE_WIDTH, 0.5f); glVertex3f(0.5f*SCORE_RATIO*SCORE_FONT_SIZE, SCORE_FONT_SIZE, 0.0f);  // top right
				glTexCoord2f(SCORE_XOFFSET, 0.5f); glVertex3f(-0.5f*SCORE_RATIO*SCORE_FONT_SIZE, SCORE_FONT_SIZE, 0.0f);  // top left
			glEnd();
		glPopMatrix();
	}

	uint16_t n = 1, m = score/10;
	while (m > 0) ++n, m/=10;

	color[0] = team ? 1.0f : 0.0f, color[1] = 0.0f, color[2] = team ? 0.0f : 1.0f;
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);

	glPushMatrix();
		glTranslatef(team ? 0.95f*field_width : 0.05f*field_width, 0.94f, 0.0f);
		glTranslatef(team ? -1.f*DIGIT_RATIO*SCORE_FONT_SIZE : (n-1)*DIGIT_RATIO*SCORE_FONT_SIZE, 0.0f, 0.0f);
		draw_digit(score%10, SCORE_FONT_SIZE);
		score /= 10;
		while (score > 0) {
			glTranslatef(-DIGIT_RATIO*SCORE_FONT_SIZE, 0.0f, 0.0f);
			draw_digit(score%10, SCORE_FONT_SIZE);
			score /= 10;
		}
	glPopMatrix();

	glDisable(GL_TEXTURE_2D);
	//glPopAttrib(GL_TEXTURE_BIT);
};


void Renderer::draw_variance(float mean, float var, uint16_t team) {
	GLfloat color[] = {team ? 1.0f : 0.0f, 0.0f, team ? 0.0f : 1.0, 0.5f};
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);

	glPushMatrix();
		glTranslatef(team ? field_width+2.5f*PADDLE_RADIUS+BALL_RADIUS : -2.5f*PADDLE_RADIUS-BALL_RADIUS, 0.5f, 0.0f);
		glScalef(PADDLE_RADIUS, 2.0f*sqrt(var), PADDLE_RADIUS);
		glBegin(GL_QUADS);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glVertex3f(-0.5f, -0.5f, 0.0f);
			glVertex3f(0.5f, -0.5f, 0.0f);
			glVertex3f(0.5f, 0.5f, 0.0f);
			glVertex3f(-0.5f, 0.5f, 0.0f);
		glEnd();
	glPopMatrix();
};


void Renderer::draw_digit(uint16_t digit, float size) {
	GLfloat u = digit * DIGIT_WIDTH;

	glBegin(GL_QUADS);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glTexCoord2f(u, 0.5f); glVertex3f(0.0f, 0.0f, 0.0f);  // bottom left
		glTexCoord2f(u+DIGIT_WIDTH, 0.5f); glVertex3f(DIGIT_RATIO*size, 0.0f, 0.0f);  // bottom right
		glTexCoord2f(u+DIGIT_WIDTH, 1.0f); glVertex3f(DIGIT_RATIO*size, size, 0.0f);  // top right
		glTexCoord2f(u, 1.0f); glVertex3f(0.0f, size, 0.0f);  // top left
	glEnd();
}


void Renderer::draw_field() {
	GLfloat color[] = {0.0f, 0.0f, 0.0f, 0.0f};
	glPushMatrix();
		glRotatef(-22.5f, 1.0f, 0.0f, 0.0f);
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, field_texture);
		color[0] = 0.1f, color[1] = 0.1f, color[2] = 0.1f, color[3] = 0.3f;
		glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
		glPushMatrix();
			glScalef(field_width, 1.0f, 1.0f);
			glTranslatef(0.5f, 0.5f, 0.0f);
			glRotatef(45.0f, 0.0f, 0.0f, 1.0f);
			glBegin(GL_QUADS);
				glNormal3f(0.0f, 0.0f, 1.0f);
				glTexCoord2f(0.0f, 0.0f); glVertex3f(-0.6f, -0.2f, 0.0f);  // bottom left
				glTexCoord2f(1.0f, 0.0f); glVertex3f(0.6f, -0.2f, 0.0f);  // bottom right
				glTexCoord2f(1.0f, 1.0f); glVertex3f(0.6f, 0.2f, 0.0f);  // top right
				glTexCoord2f(0.0f, 1.0f); glVertex3f(-0.6f, 0.2f, 0.0f);  // top left
			glEnd();
		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
	glPopMatrix();
}


void Renderer::render() {
	glClear(GL_COLOR_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.5f*field_width, 0.5f, 1.5f, 0.5f*field_width, 0.5f, 0.0f, 0.0f, 1.0f, 0.0f);

	draw_field();
}
