if (typeof T === 'undefined') require('../setup');

T('dividedBy', function () {

  Decimal.config({
    precision: 40,
    rounding: 4,
    toExpNeg: -7,
    toExpPos: 21,
    minE: -9e15,
    maxE: 9e15
  });

  var t = function (expected, dividend, divisor) {
    var quotient = new Decimal(dividend).div(divisor);
    T.assertEqual(expected, quotient.isZero() && quotient.isNegative());
  }

  t(false, 1, Infinity);
  t(true, 1, -Infinity);
  t(true, -1, Infinity);
  t(false, -1, -Infinity);
  t(false, 0, 1);
  t(true, 0, -1);
  t(true, -0, 1);
  t(false, -0, -1);
  t(false, 0, Infinity);
  t(true, 0, -Infinity);
  t(true, -0, Infinity);
  t(false, -0, -Infinity);

  t = function (dividend, divisor, expected) {
    T.assertEqual(expected, new Decimal(dividend).div(divisor).valueOf());
  }

  t(1, 0, 'Infinity');
  t(1, -0, '-Infinity');
  t(-1, 0, '-Infinity');
  t(-1, -0, 'Infinity');
  t(1, NaN, 'NaN');
  t(-1, NaN, 'NaN');
  t(0, 0, 'NaN');
  t(0, -0, 'NaN');
  t(-0, 0, 'NaN');
  t(-0, -0, 'NaN');
  t(0, NaN, 'NaN');
  t(-0, NaN, 'NaN');
  t(NaN, 1, 'NaN');
  t(NaN, -1, 'NaN');
  t(NaN, 0, 'NaN');
  t(NaN, -0, 'NaN');
  t(NaN, NaN, 'NaN');
  t(NaN, Infinity, 'NaN');
  t(NaN, -Infinity, 'NaN');
  t(Infinity, 1, 'Infinity');
  t(Infinity, -1, '-Infinity');
  t(-Infinity, 1, '-Infinity');
  t(-Infinity, -1, 'Infinity');
  t(Infinity, 0, 'Infinity');
  t(Infinity, -0, '-Infinity');
  t(-Infinity, 0, '-Infinity');
  t(-Infinity, -0, 'Infinity');
  t(Infinity, NaN, 'NaN');
  t(-Infinity, NaN, 'NaN');
  t(Infinity, Infinity, 'NaN');
  t(Infinity, -Infinity, 'NaN');
  t(-Infinity, Infinity, 'NaN');
  t(-Infinity, -Infinity, 'NaN');

  t(1, '1', '1');
  t(1, '-45', '-0.02222222222222222222222222222222222222222');
  t(1, '22', '0.04545454545454545454545454545454545454545');
  t(1, 0144, '0.01');
  t(1, '0144', '0.006944444444444444444444444444444444444444');
  t(1, '6.1915', '0.1615117499798110312525236210934345473633');
  t(1, '-1.02', '-0.9803921568627450980392156862745098039216');
  t(1, '-0.0001', '-10000');
  t(1, '8e5', '0.00000125');
  t(1, '1e-14', '100000000000000');
  t(1, '-94.12E+0', '-0.01062473438164045898852528686782830429239');
  t(1, 'Infinity', '0');
  t(1, '-Infinity', '-0');
  t(1, Number.POSITIVE_INFINITY, '0');
  t(1, Number.NEGATIVE_INFINITY, '-0');
  t('0', 0, 'NaN');
  t('0', '0', 'NaN');
  t(3, -0, '-Infinity');
  t(9.654, 0, 'Infinity');
  t(0, '0.001', '0');
  t(0, '111.1111111110000', '0');
  t('NaN', 0, 'NaN');
  t(0, 'NaN', 'NaN');
  t('NaN', '0', 'NaN');
  t(-1, 1, '-1');
  t(-0.01, 0.01, '-1');
  t(54, -54, '-1');
  t(9.99, '-9.99', '-1');
  t('0.0000023432495704937', '-0.0000023432495704937', '-1');
  t(NaN, 'NaN', 'NaN');
  t('NaN', NaN, 'NaN');
  t('NaN', 'NaN', 'NaN');
  t('NaN', 4, 'NaN');
  t('NaN', '4534534.45435435', 'NaN');
  t('NaN', 99999.999, 'NaN');
  t(Infinity, '354.345341', 'Infinity');
  t(3, '-Infinity', '-0');
  t(-Infinity, '-Infinity', 'NaN');
  t('-Infinity', -Infinity, 'NaN');
  t('Infinity', '-999e999', '-Infinity');
  t(1.21123e43, '-Infinity', '-0');
  t('-999.0', 'Infinity', '-0');
  t('657.342e-45', '-Infinity', '-0');
  t('Infinity', 123, 'Infinity');
  t(-0, 'Infinity', '-0');
  t(100, 100, '1');
  t(-999.99, '0.01', '-99999');
  t('10', 4, '2.5');
  t('03.333', -4, '-0.83325');
  t(-1, -0.1, '10');
  t('99999', '1', '99999');

  t = function (dividend, divisor, expected, sd, rm) {
    Decimal.precision = sd;
    Decimal.rounding = rm;
    T.assertEqual(expected, new Decimal(dividend).div(divisor).valueOf());
  }

  t('999.5', 1, '1000', 3, 0);
  t('-999.5', 1, '-1000', 3, 0);

  t('999.5', 1, '999', 3, 1);
  t('-999.5', 1, '-999', 3, 1);

  t('999.5', 1, '1000', 3, 2);
  t('-999.5', 1, '-999', 3, 2);

  t('999.5', 1, '999', 3, 3);
  t('-999.5', 1, '-1000', 3, 3);

  t('999.5', 1, '1000', 3, 4);
  t('-999.5', 1, '-1000', 3, 4);

  t('999.5', 1, '999', 3, 5);
  t('-999.5', 1, '-999', 3, 5);

  t('999.5', 1, '1000', 3, 6);
  t('-999.5', 1, '-1000', 3, 6);
  t('999.4', 1, '999', 3, 6);
  t('-999.4', 1, '-999', 3, 6);
  t('999.500001', 1, '1000', 3, 6);
  t('-999.500001', 1, '-1000', 3, 6);

  t('999.5', 1, '1000', 3, 7);
  t('-999.5', 1, '-999', 3, 7);

  t('999.5', 1, '999', 3, 8);
  t('-999.5', 1, '-1000', 3, 8);

  t('-2.8', '3', '-1', 1, 3);
  t('-3', '-0.0047', '638.297872341', 12, 0);
  t('15', '-3', '-5', 1, 0);
  t('3', '0.38', '7.894736842106', 13, 2);
  t('-2.9', '-2.2', '1.3181818181818182', 17, 4);
  t('-1.20', '0.00035', '-3428.571428571', 13, 4);
  t('-6', '-1.0', '6', 1, 1);
  t('1.1', '-1.5', '-0.7333333', 7, 5);
  t('-1.05', '-2', '0.525', 4, 2);
  t('-2', '-1', '2', 13, 0);
  t('-5', '1', '-5', 16, 6);
  t('0.000033', '-1', '-0.000033', 11, 1);
  t('6', '-3', '-2', 20, 6);
  t('-1.20', '1', '-1.2', 13, 2);
  t('-1', '0.0019', '-526.315', 6, 1);
  t('-7', '3', '-2.333333333333', 13, 1);
  t('2', '4', '0.5', 8, 0);
  t('33', '-2', '-16.5', 14, 3);
  t('-2', '-1', '2', 4, 1);
  t('-10.2', '12', '-0.85', 16, 6);
  t('-1.0', '10', '-0.1', 10, 0);
  t('1.3', '-3.0', '-0.434', 3, 3);
  t('1.4', '6', '0.233333333333333333', 18, 5);
  t('55', '2', '27.5', 4, 4);
  t('1', '7', '0.14285714285714286', 17, 5);
  t('-1.4', '2', '-0.7', 12, 5);
  t('-5', '3', '-2', 1, 4);
  t('-1', '2', '-0.5', 15, 0);
  t('-8', '-3', '2.67', 3, 2);
  t('-16', '7', '-2.285714285714', 13, 4);
  t('1', '1', '1', 13, 6);

  Decimal.toExpNeg = Decimal.toExpPos = 0;

  t('-9.0201582007891419174890e+493625', '-5.50622769083719199346231074389e-8', '1.638173847369118854685833779858335057671769579218908745636679e+493633', 61, 4);
  t('5.8729543161021068260124718708021832883057695897e+49274', '-8.708403405379587884587853336e+41461', '-6.7440081065538470526693072631940999905631492068918496007e+7812', 56, 3);
  t('-1.11303135570566757735452613975119540626335099312445395511e+909913', '-4.9428912981238120390359176251627374079558e-50436', '2.25178198057510227e+960348', 18, 1);
  t('9.34964786072062858037053820543908584e+34427', '-5.5139174707961944758881146494671658725524731327174937382473e+727', '-1.69564523049899134866471123135148425607621655195972485244321739609622142643026524021192529472e+33700', 93, 4);
  t('-8.95495805776908286477460208890137140530903524050e+3546735', '-514449923650777303707976731752571405.60601209', '1.740686050494577121230344265740919289154954933388084539546267265149437887923633011167e+3546700', 85, 0);
  t('-112977.156416947803174690696837418982360160807579018', '-6.7793027141628576643753135768314185053384986437525e+32928', '1.666501131169782697443349597254128378095825062301415158287415940806830463694e-32924', 76, 5);
  t('-7.519050808224891132821326159698e-555', '0.0417050002083', '-1.80291e-553', 6, 1);
  t('0.0312143762310912553081058912', '-4.08382620086677699331679658629041009e-90', '-7.643414458839e+87', 13, 1);
  t('4.0891643182290652875211844988576636684270563255557e-173776', '-4.3512242516716750610555570756791193e-72054', '-9.3978e-101723', 5, 0);
  t('1.6121785095818917662044501035572e+171567', '5.6108971779838795537551844770788131976757e-524750', '2.8732989724e+696316', 11, 3);
  t('7.5653760375e+30', '8.15774224e+1590', '9.27386011340805e-1561', 15, 4);
  t('-8.660566651142957641766979855e-18592', '6.487325740700605731241602360168e-47415', '-1.33499796e+28823', 9, 1);
  t('-4.500945343043809676875681e-43152', '5.940827660904742303158570004409569371251583570495996069e+867097', '-7.57629340548545e-910250', 15, 0);
  t('8.96986622064415401423819967094260833389514278486115338442e-7572812', '8.4104122599652535974085220853977252989342871715575506e+4831', '1.066519207785090396984112228095053363096953582614945755253738194960648243775759819978376352442801954e-7577643', 100, 2);
  t('-6.856960316080e-75', '-2e-97', '3.428e+22', 4, 3);
  t('-7.24821889609458e-116548', '-1.25771065437743895862578613162692675e-296408', '5.7630257570509453279427403558712890904042059962e+179860', 47, 6);
  t('1.66634161834784366386855049316805612010518245355860875e-1433178', '-1.2323759671622999856410026493670230698834833365211381e-119', '-1.35213738562656649534995020628832773189172872015634e-1433059', 52, 2);
  t('-9.89858514790913976013855098325866480343405831726e+1326732', '-3.128944043635047374403186608489765604611130599212469197e+459', '3.1635545442383396285744071213319905e+1326273', 36, 0);
  t('-8.070e-906304', '-1.5055451435372855159976696184756e-124', '5.36018467107502097209360022333300020062468708142031209526934182877919e-906180', 69, 6);
  t('-9.046986850521113233994088030748718263e+649192', '7.42739066e+75', '-1.2180572242205330874563272306385885272661826030839207264749e+649117', 59, 3);
  t('-7.0140966397407758056324106e-567327', '53259.7776732322775987808330561104626988', '-1.31695942907887432785089328517373629792e-567331', 39, 1);
  t('7.40231507068915012684011850565156286998231535809e-895430', '-8.86406984919e-643', '-8.35092141265738645015908443920311470645480941163030158470835428757522e-894788', 69, 2);
  t('-4.921794410e+414322', '-1.0785013517e-841504', '4.5635495980064982611184983274230976561881299309270026573671840860243587583442e+1255826', 77, 5);
  t('601866548.752324072028390640262687694145312', '9.1268385395129583279994237e-1284', '6.594469115966654151749202035640916912881336984e+1291', 46, 2);
  t('2.867568612241923911789069819170160836469e-1906692', '-7.12441896102389693918811169596243150730e+81791', '-4.0249859362984554655443106786916039152860369641143618688585255330551064941359213504322433e-1988484', 89, 1);
  t('-5.23144143309771e-1367', '-7.6883670574418716341842245776e+81971', '6.804e-83339', 4, 5);
  t('-1e+46', '9.71058916684636392105974344694162750232811143394e-47', '-1.029803632733401478090931970815421478665673365628839811617483095659275091488624129e+92', 82, 2);
  t('7.2e-59409', '5.7831139640342802478088151675935006389840e+703', '1.245003996943076834455034433890469750759315904851845063541633723247163682755452146320767e-60112', 88, 1);
  t('9.5511191557227891017e-976054', '9189124852', '1.0393937735696344962122e-976063', 23, 6);
  t('4.23712926488581167975667516493342425898064e+57103', '-0.0001397286766', '-3.0323977640004440432499416980332469984e+57107', 38, 2);
  t('1.2743888136325381752424465340840098569178e-7', '-18573238529588601.01391', '-6.86142490229874847637478954198611190873400267067557244764e-24', 57, 3);
  t('-1.104892840718228416873e-7', '3.5597797377667191893695e+934', '-3.10382361300645918144338685515593065286843980322155334919919021986707800416857125298e-942', 84, 6);
  t('-1.55519573186342035670507252118450593339698096309942436625e-817998', '5.0563859624221211468025641352239672653831983e+64587', '-3.075706133632344533175127773502161481034704186956042445418520782007015e-882586', 70, 5);
  t('-6.202151355028114221575433171095985096e+308', '6.033804697384754103136626236314', '-1.027900581156749571100486532662322045801648336202116385320823112202169619166115945348630645e+308', 91, 3);
  t('8.024875460866071e-302608', '5.359320147511614393506518038e-445413', '1.497368181035443540224019658376422e+142805', 34, 2);
  t('3.55055235944317737144206844338561241970e+384', '-0.0022624534102156295443300961', '-1.56933722630990306246023742223862701e+387', 37, 1);
  t('8.124983863334591032859756583631905836816939039683944086589e+609557', '-1.60542904153166299378981482239854e+559466', '-5.06094237312596070034955736800076847046972453544295277599851337171871887357766954662496371928334e+50091', 96, 6);
  t('1.2361212094491178193e+83711', '-6.1097447943101726e+77', '-2.023196141678915138239321117934234676090045505345373146119122780711362684355605637208296539355003037e+83633', 100, 3);
  t('-9.7913464098919312889521180664170915225195748592237470281501e-2442', '-7.117703009727629530310739342559e-531056', '1.3756328968081815297848119554475753176429302386798137109462696314553328545836837194134940777e+528614', 92, 6);
  t('-1.8008395847460187443885218251028268177940e-181136', '-1.9315535289572721940531408208e+1701436', '9.32327040254577916615185349472487605397250031648341346028e-1882573', 57, 3);
  t('-1.69914659289020442692747108811266537843629740471379191228e-750409', '1.99579900340005978891346263731685e+87', '-8.5136157999654481886972931e-750497', 26, 0);
  t('-3.468594080438267571477376858183887124615200e+92', '5.6257641868735912825310409703621015617e+971497', '-6.165551852549068675875315487455698620448242921611056694146328281821305148315378561e-971406', 82, 5);
  t('-1.8252099405739206816451704100248812959e+1186', '-3.42243088142742109427052805672e-9', '5.333080502746823979506640819479922185269986904173244526721149958304591733214947e+1194', 79, 4);
  t('-2.26794496381269349400848879905354658898089516910667e+492', '6.9657557700813802540e-7189957', '-3.25584909760078670021134909504190734019544459726e+7190448', 49, 0);
  t('-0.46804364341836888128551806365696954134445', '7.987261632972183047981868785502422780e+821', '-5.859876199450382023516232392475110813719370468990469393875819714917792471118581757029002014878e-823', 96, 2);
  t('-9.96705038725844823301661858980473138e+947', '-7.2408733e+507', '1.3765e+440', 6, 4);
  t('6.4e-65903', '2725.9585677573990738059898886877', '2.347797973050329194430525714895592743169714777546007978234095454030731491e-65906', 73, 4);
  t('3.248366411826949431866026704463662e+52511', '-9.2033435415451915214110831e+8071781', '-3.5295503174073259785984985357981547e-8019271', 35, 3);
  t('-9.98470001351862389956296e+451', '3.7474300286329393632461064424e+815', '-2.6644126607378010385763797854701476440577291827637607081033950365026e-364', 68, 0);
  t('-1.53740039467703159465920401080173826327989263865383e-74', '-1027.743193807096525634910', '1.49589936857864103759617619966197637279298146868e-77', 48, 4);
  t('-7.1898403140998064976226699627450254188793061616078584e+4075', '3.312101273050083107245022e+70', '-2.170779128223561156107980514002547628515920675156954075597305599528830244568459314942e+4005', 85, 4);
  t('4.06e+13901', '-5.85611e-50381', '-6.9329298800739740202967498902855308387308298512152264899395673920059561722713541924588165182e+64281', 92, 3);
  t('3.78987311925956596100530099233127e+533', '-7.7787839e-446', '-4.8720637672677421479793274528827957285199811245559861869925451972e+978', 65, 5);
  t('5.71355915176845523331889400e+85', '7.9368447699694051170227763009331626968309196669062001362222e+28146', '7.19877900773216182411356396344316798e-28062', 36, 3);
  t('-7.6300279510508555176447528888582183176281948759413355783e+924643', '-6.525229144297962820139259674034141343420e-98555', '1.1693118789120709003842725006130814665259894299168791890471624080948019031651507617258471260255226e+1023198', 98, 5);
  t('7.28439100831730318544400863622338e-975118', '-90382941.9311', '-8.05947544158308297233205000580769151550908738308801809853417304107e-975126', 66, 6);
  t('5.9080280425535882593423530335827988725154144673935e+150783', '-9.7116925492708374913103514832318668414602235998045246e+240', '-6.0834175017e+150542', 11, 5);
  t('-8.32876439894620965060923423888278658060936060382855e+5391166', '8.04186747558480860e+491', '-1.035675410497925143157313386715336111549526162006770420515223349892e+5390675', 68, 5);
  t('6.615091161705197007910219512584837928991296e+289223', '3.671292452338065773949802592091439062162827897496531e-514040', '1.8018426065437446911933669148638e+803263', 32, 2);
  t('4.22074155824438744041754715460773774e-26794', '-0.392620260033033013334236126522330408072615', '-1.075018787336465097611404243630364449447595646694657386098906315606038505272594760203526802864099342e-26793', 100, 0);
  t('-4.67699736667236482648690536440818e-535947', '-0.000003272760314495064262408864180258266590034864302365641', '1.4290681006971182312140595347675928317716659547290799132515275441516238979e-535941', 74, 6);
  t('-0.000084398040461600827616061519', '-1.189305971339233178095314502175250728139881670533927588e-451', '7.09641105783428769600139659965663184560402725e+446', 45, 5);
  t('5.36563080174153764198593156640406584047160058605498e-827949', '-5.117680469211393e+56', '-1.048449748674588988737499894175e-828005', 31, 4);
  t('85.589843957916002336981173213', '0.0000578255253220352633511490632580338770863578897775', '1.4801394968962e+6', 15, 1);
  t('-351607.1187872102002591263170005488', '-9.43211176998759831430041388795767495627847463971963992990e+9816897', '3.7277666694536265675007568840266918599145597645718155579541605e-9816893', 62, 1);
  t('5.54766258186312012212070335772068830425136308583014956e+69623', '6747132.4960288947760609757117734740909355201', '8.222252319971874031e+69616', 19, 3);
  t('-7.40794403e-1867427', '1.07829970890779e-80', '-6.870023212288082e-1867347', 16, 1);
  t('2.8138856089460929272361416217404739394197772e-5864', '4.120358854405098750175445168459012485500020673033832e+13388', '6.829224609739396965225736675346830773941251044978533929261416840376667913341292696644e-19253', 85, 2);
  t('1.25611e-7382', '5.22783672860544552165472229850258202818243975e+561', '2.402733798335500653391738916406216392904413019e-7944', 47, 1);
  t('-4.298240874909906897e-28', '-4.25064103499e+369', '1.01119827327880178894776703201179105690352e-397', 42, 5);
  t('-3.3734578706259646249822280062e-9836', '2.04159630139282492575938762375186e+823662', '-1.6523628439e-833498', 11, 0);
  t('-1.4929903958127657150797046e-31761', '-9.34820994328909660952679308089769623534838087e-504367', '1.597086934151e+472605', 13, 1);
  t('-6.17416836518150351171259742611706e-97', '-9.88775533288148875e+37187', '6.244256817975114615221762745361996320156750169699731652e-37285', 55, 3);
  t('-9e-296674', '-9.429077635688888467419778561698443234851940034555559e-7593', '9.54494209056e-289082', 12, 0);
  t('2.6439727192e+992732', '-2.01160292191790849717605363585281e+3688', '-1.3143611447328658640835497026217432813435045010572027729281904104297711381081829e+989044', 80, 0);
  t('-4.53845718907359787719378104959276674317e-49650', '5.08475248507297756177418035662893e+69', '-8.9256206715e-49720', 11, 2);
  t('-5.693328123037558e-50195', '2.7199658e+64302', '-2.09316165778171107886724163958238005e-114497', 36, 3);
  t('4.978838741243446e+7675146', '-2.79566412024916162230460222172486e-50', '-1.7809144901e+7675196', 11, 2);
  t('-1.88784650527777086088949271343154160454393e-1404757', '-0.0000502787620604091848652480080339383261497', '3.75475932165861872458913296058527751903692013400759286e-1404753', 54, 5);
  t('-5658507039746715423.167542720210950494520611683594823141492', '5.7725075471049662533145488840960912984928824220440e+302000', '-9.8025112891962792177142779236158942424695123056e-301983', 47, 5);
  t('-5.67859887941334558531160e+4269739', '9.0616208329978377544345050634619764178276772330038011031e-903', '-6.266648079927116266831922933734264307785369125e+4270641', 46, 2);
  t('-5.153438159204456445e+441963', '-3.910302066e-1945068', '1.3179130594573499747116467395693005779160182148444794852838358702900268473e+2387031', 74, 5);
  t('6.50766000334324292674204104205256939890391290244703e-570636', '-6.13825054376718197194260486288608e-36', '-1.06e-570600', 3, 5);
  t('0.0003561032581306', '4.77053286770261875972371870e+87', '7.46464321714423724166117282096633e-92', 33, 3);
  t('423561.59245453820356744150114646992866121611599555557', '-7.96e+8386897', '-5.32112553334846989406333544153856694298010195974316042713567839195979899497e-8386893', 75, 4);
  t('-5.1558130542057773853e+483', '6.2', '-8.315827506783511911774193548387096774193548387e+482', 46, 6);
  t('9.80010135321953141397e-9479241', '1.48559844e+2465', '6.59673643250428521853455904275182195264017643960369263715705032646641712951717962224031414573e-9481706', 93, 6);
  t('2.921469205835105144020568210975511896e+994', '-2.850750914899798e+8528563', '-1.02480689932104883493349188185159896705506939026865209759783571883322480129714412160052799e-8527569', 90, 5);
  t('6.02475139808495394997772158116702e-21', '2.37617147374471492952579010471604836012667268737402989000675e-7788231', '2.53548679657797536354423e+7788210', 24, 2);
  t('87513.32883116404188225508498894548197167557912759240052210', '3.7772896063968138047935970431e-5045', '2.3168286774453519584522693858683014239766232911352581157877601882e+5049', 65, 3);
  t('5.7718214554244629387781323483e-928', '-8.80697537569701968499916566682238518251640403050597553611e+765307', '-6.5536931911401621494952381236078093131919781294446e-766236', 50, 3);
  t('-7.31783566924948556509724994049448360187e+43', '4.193922413377111528576823606882686875e+132239', '-1.7448667256953e-132196', 15, 2);
  t('1.5763909385752914105437373419998105285263445570305756313e-939065', '5.2696546007e-743', '2.99145e-938323', 6, 5);
  t('-7.3695076326257297560e+939', '-7.566e+873913', '9.74029557576755188474755e-872975', 24, 4);
  t('-7.615009147460135e-2238951', '5.098853922503091887250195828406763584e+2029', '-1.493474663757739247e-2240980', 19, 0);
  t('4.7118761952579331686862200927113271192e+972298', '-8.1045590739e+3987', '-5.81385878280794397787160794e+968310', 27, 0);
  t('3.05259816713008214309752e-34', '9.072438477569600795244081994373265659e+3600595', '3.3646942601784797867441e-3600630', 23, 5);
  t('3.5933388527257947e-69', '7.1641505002922391424671237937068824e+93521', '5.0157221747075465058e-93591', 20, 2);
  t('9.39168343132314935e+371121', '5.3947954122933792607331297865723793172399569985e-387897', '1.74087851597149903151101989151427616342421252127054094524491742593e+759018', 66, 1);
  t('6.8421850057907933405544695887211983810452221801e+684439', '9.162088e-434522', '7.46793198863708069662119550556728813458812246738952954829e+1118960', 57, 2);
  t('2.54275263810343519553437544378832436e-303', '-0.08770599033318', '-2.89917784229e-302', 12, 2);
  t('31981.165449494238805174660659817376468695645961', '2.23651670491568483146967340797091924533e-989818', '1.429954239966e+989822', 13, 3);
  t('-4.3304985248849022107e+84598', '721382.36921875', '-6.00305567430874344247379626761969984667e+84592', 39, 3);
  t('-5.69941467419121688624215027573910521249071692951357140749241e+2238', '1.47308952694108000853902e-188201', '-3.869021244096577970516432071542636232690279162375e+190439', 49, 3);
  t('-807045896.37201888946402216019903869620386', '-9.08231770105120e+35', '8.88590250788749971057617554208065127368874350003175872528293248627303374010246161422038e-28', 87, 4);
  t('-3729', '1.0546094247819167974007e+3449', '-4e-3446', 1, 5);
  t('-5.47050611316e-521048', '1.682277204e+919126', '-3.25184583144360315542859843686e-1440174', 30, 4);
  t('-4.3283158011e-852855', '8.1631725039603251121269990518659693605483302e-6995', '-5.30224713369726996773806997000061981936102436057537e-845861', 51, 2);
  t('3.830790634e+338750', '-1972.99790017035104904780622921', '-1.9416090780782102488343413702801e+338747', 32, 6);
  t('-675363513.402269898572629297102147534327342971037825167', '2.3e+4480635', '-2.936363101748999559011431726531076236205839e-4480627', 44, 6);
  t('-6.9843101214254330530310114e+875601', '1.78702225591614033015555756095728421e-37748', '-3.9083509443170505610410562761111795229709958845170193696171800343480046642337394354637126600389418e+913349', 98, 2);
  t('6.16537500715193614011018728455679e-885', '7.872343857695937901e-745', '7.831689162211476312583530674208619972304918867871e-141', 49, 2);
  t('3.727006137610643885052534419530511e+78', '-1.425386648485847138744055919504080772e-886', '-2.61473344202413429907946769817909664196987836016435350482399200706130745727594349802635891586e+964', 93, 4);
  t('7.43959327574e+5181', '-7.9074313981397299324949e-52', '-9.4083563943282623716203312085417443848809831081928646668382343690834056267e+5232', 74, 6);
  t('-3.4490340985404845139552633843307403193543e-6651', '7.2842692393679412330016944012164504254989382282e-426825', '-4.7349074906514005968e+420173', 20, 6);
  t('-3.12646845651434317784659897194601292447905599816047557896e+83609', '-4.44821508711659247554239e+84531', '7.028591e-923', 8, 1);
  t('-4.9158931676508508586366306659252842802577412066912290e+6841', '-9.97885279829e-4348', '4.92631093675742970055851625020542497534673412306023848256715519495285e+11188', 69, 6);
  t('4.5204228951407159990066208e-3745', '-0.004', '-1.1301057237851789997516552e-3742', 69, 0);
  t('-3.09307e+5347', '-0.8984736838504547635274923785', '3.442582744042642266786782278260624614507789e+5347', 43, 3);
  t('1.278957083641480297045115728536640377e-882', '-9.1829578959430558e-90519', '-1.3927506780865362298960771170837187909688134215939e+89636', 50, 4);
  t('-1.280e-30680', '-7.809867371408833685193380117868583027e+33846', '1.638952288339691587843478189801175e-64527', 34, 0);
  t('-4.796625599778919175150221338154194105391e-3693', '1.0644936863715949e-3575', '-4.5060160160541399e-118', 17, 6);
  t('-2.338716751466980568104931647443388782971007e-594', '-8.76079791822909184588455192060496995121228203764073135e-7', '2.6695248233048262067724344845187321954535e-588', 41, 5);
  t('-2.88857445130347e+34', '-4.5727677050043269726429416e+57466', '6.3169061663514715799083365164448167754425592465222489e-57433', 53, 0);
  t('9.0464589776483057165017421e-30', '6.06745997e+141768', '1.490979589874130758624805908690650990813211743364826847e-141798', 56, 4);
  t('4.0891e+8682457', '-0.285100211208484985423775082', '-1.43426761511929129076674694508290636841735067147441019762532785721934538893495633683716943287921582e+8682458', 99, 2);
  t('-8.36533512511e+2210', '-2.2074165483312369355092715221e+9854', '3.7896495482169086724607338870566377482914270703864737066601275861373795191e-7644', 75, 5);
  t('5.52706150135816541190537725676638784816e-39', '5.506157558676409241502564869727746514208939474698e+18904', '1.00379646649391942543124561396333984453880182681374708614940506997100647153719275030013e-18943', 87, 5);
  t('-1.288279714202288615454e-596927', '-3.77546319534770573354581697206296e-42949', '3.412242809808778910720913252239877519936414468601131742686648528257078142609085752972025220629296596e-553979', 100, 5);
  t('-6.153052379002130168510557374994653393268', '1.7358896156255864099559044599825626297210e+3157200', '-3.5446103966608902581464527977e-3157200', 29, 5);
  t('3.42374045880379367998282136916009048354975475e-459', '-3.71961470952892229383923198866553177381531730879349520967e+9323392', '-9.204556724740449869859132948088048642326875592013709820825989488772e-9323852', 67, 5);
  t('-8.029527535353e-8923', '-382173906.1', '2.10101406903798029867638836161765885669398400614667187504249129058997259467788e-8931', 78, 3);
  t('7.54405949614753081e+2945477', '-1.2969572371386e-39', '-5.81673726790834124060476166960328240382299131951266748135316067254764231735799071285353603929078e+2945516', 96, 1);
  t('-52465', '6.04671115345965326980458938153601846588748169e-44', '-8.67661753116517096859514986859114860862090070653645935252959574e+47', 63, 4);
  t('1.4633750786e+28385', '6.4e-67', '2.2865235603125e+28451', 20, 4);
  t('7.4803238e+6937754', '-398287514481567281059186496112.143318', '-1.87812159006209291405314267025131725835563091083703200385262e+6937725', 60, 6);
  t('-3.99879850022268386238702401202296926182374685e-101020', '4.941987958200394155523261928737367666185011455660081e+1521', '-8.09147762812200559345178485001850079766691e-102542', 42, 1);
  t('-6803066.78655613559643867402367500170440266090475412341', '-4.35752016541514522894124887798932462e-9', '1.561224395597949169169135e+15', 25, 4);
  t('4002943.595339596145529525507585154374501889180', '7.48365108818012585828282380463973422306674153125104052e-5682614', '5.3489179923e+5682619', 11, 4);
  t('3.76872899581827988357416381361938138263757177508e-8', '-1.3878956626098432e-4521', '-2.715426740891632926325292842225052328752037516453023854502281226161793716837461e+4513', 79, 3);
  t('-1.8143315369896313220192384990961747995084538536e-652154', '-1.64972e+37', '1.099781500490768931709161857221937540618076918264917683000751642703004146158135926096549717527822904e-652191', 100, 0);
  t('1.077950051217448803307899727500419929851e+67383', '9.63133018252238196604554120627877300948e-5273897', '1.119212020343321654951674321950063587369212979650466228152527216933560442214435806e+5341279', 83, 1);
  t('-4.9692553578428e+924108', '2.1462368e+32185', '-2.3153341503802376326787426e+891923', 26, 6);
  t('5.930990738383053010721102273e-97036', '-3.65966511964763315813782962456128293763264222893735645e+500', '-1.6206375568467619650484357428145577548720152013649725255302838490585441e-97536', 73, 0);
  t('4e-75671', '-1.70839225333645877614084793215132655045157016180043878649e-12843', '-2.3413826609129567713111285198617587201274761608761376874e-62828', 56, 1);
  t('6.91439488419339340315975001e+5596001', '-5.16903704616137e+77', '-1.33765628345e+5595924', 12, 1);
  t('-3.7272665275459591457414082845e+890800', '6.572249044180767e+314787', '-5.6712192470036931497313123e+576012', 26, 5);
  t('-5.7045884550662551247342877917788744897456312e-51697', '-3.44612644750133357600866e+9136468', '1.65536248944157397260819252517234149348139815239687213295268100601194e-9188165', 69, 1);
  t('-5.798080270240466658842194604602234815273651577331e-604', '0.000005507441779976069848143514575', '-1.05277195871976328938e-598', 21, 1);
  t('-1.12716558087499949577479980549898647746137202857e-42550', '-3e+6', '3.75721860291666498591599935166328825820457342856666666666666666667e-42557', 66, 6);
  t('9.029304e-44495', '-3.64145555513779926589009748930107e-983355', '-2.479586490424243225977952663944442587563972524447300319836489006379990970201351934707743567069631277e+938860', 100, 3);
  t('-4.520055620360621304275037267801651847705e+3563240', '-1.756992768930296764206837935869004010161438300752191e+31082', '2.572609119565443404701767686555306e+3532158', 34, 5);
  t('-6.511633724726488e-908197', '8.741385376254644142100e+9923', '-7.449201065e-918121', 10, 3);
  t('-4.727173765422559931e+2550', '-5.0e-7235', '9.454347530845119862e+9784', 87, 6);
  t('1.70316439e-9459', '-7.8361e+58', '-2.173484756447722719209811002922372098365258227945023672490141779711846454231058817524e-9518', 85, 4);
  t('-5.406631153924624319629205640970181e+2086152', '3.480623868294410918927077008146482778988703493268074099466e-189433', '-1.55335116878745163354434679883585669329899340343081904982491891350342123010900374959249020654229e+2275585', 96, 1);
  t('-9.7492472756056798812336104410110379537907097266e+46638', '7.75465e-1490378', '-1.2572130625631949709185599e+1537016', 26, 0);
  t('-3.1908288340733e-2109630', '-8.088574102565401236091654734238720e-89', '3.9448594939141e-2109542', 14, 5);
  t('4.73970296075144109211823587827902601864e+995', '9.0e+70025', '5.2663366230571e-69031', 14, 1);
  t('-1.8152795e-62017', '-0.0000180100506137', '1.007925818164631713535804587054268307128e-62012', 40, 4);
  t('-2.99866476344257805037744687194745964863e+3924687', '9.10646106540838364998613834e-20', '-3.29289802251858799226153509010269776429660641640999375e+3924706', 54, 3);
  t('7.18895376979e+443864', '-2.96156741318470092385755745122158310400744307157681974007161e+125453', '-2.4274152051326795744379402883333820009191903153566464579425606259197487511e+318411', 74, 3);
  t('-7110585455931675754137078581369901739.4931396', '-7.18603961126806198e+4805', '9.894998971035e-4770', 13, 4);
  t('5.41147077742840731903379861397587833933128736130e+650796', '5.508976354152810651962713290752845896831e-59274', '9.82300599883515369313243246351436879284328663208903699032e+710069', 57, 6);
  t('-2.36680800333145618131527e+5696847', '-1.291231267094664e-67', '1.83298535564190179e+5696914', 18, 3);
  t('9.45737237992012431716214361370012434470559418118050763503e-5773252', '-4.821896580749656394283744178082087e-41286', '-1.961338701803885270803274517454726731856417298e-5731966', 46, 2);
  t('4.3048259706272995008493754e-75086', '8.83757752624666508015471400935746193366955380200151811e-5510566', '4.87104747635019249417585435907066331197959e+5435479', 42, 0);
  t('6.02391563857731888562808414061253329299057097272948390377', '-0.00000859431905432706316128773457656901972898376', '-7.0091831598274223499618905455260634629806935712003129886689780108271e+5', 68, 1);
  t('5.155706593542208005e+69', '-3.752836823906004493059531004405347243836788044737e-44000', '-1.37381581866276755530952503327693855569871767143331967690218722410662644294938506519268901862e+44069', 93, 2);
  t('-9220750598.310382412855687591895656971256794', '6.208189361093807955850040654218e-48438', '-1.4852560162059550267737769309184427579439118150391115871709338604990312946561e+48447', 78, 6);
  t('-4.566238640798811448714676926894105655e+575', '2.880e+191881', '-1.58549952805514286413704059961600890798611111111111111111111111111111111e-191306', 72, 6);
  t('8.91604e+170', '-6.264368339072541e-53173', '-1.423294339891905e+53343', 16, 4);
  t('8.3087090877835785554965e+35979', '1.3611234824345e+2711755', '6.10430221432e-2675776', 12, 2);
  t('-0.902795617323001343413834248792', '-1.9622832254556375292432978323544368070474621301919288930e-7605547', '4.60074063525347791647263727570223375233476678038450676053498613441846299e+7605546', 72, 5);
  t('5.130267430039513e+89942', '3.306068983128470569399947418640252356869061293022225007324e+38394', '1.55177265090967277132534306390671014600343477066067674e+51548', 54, 6);
  t('-5.42709593652831824838271990398766047426230233281511774e+441', '6.060905321454542443148625e-79491', '-8.95426615115954695687550461846409521895630824450275734448571400865841848473595221582490459e+79931', 90, 3);
  t('-2.67313108587637342e+66', '-4.2978091362412049644038575659830e-8788943', '6.2197529046491139763897883718412103051148839554074761661717609361495199609413826e+8789008', 80, 2);
  t('-5.062479752314056826234932702569182e+978760', '-9.0280152787587150771', '5.60752235790423930474788883129459576987903407906072606460557140943628831849657297853921988860216893e+978759', 99, 2);
  t('-9.682208878743727894539541638693047e-8149861', '2.68789960374616143221198636526210862719600140e-805', '-3.602146771125493149886358094423260127318575198065409723915337473739698227e-8149056', 74, 1);
  t('-7.74270044930678819329480422147363893912745e+46050', '-1e+850', '7.742700449306788193294804221473638939e+45200', 37, 4);
  t('3.837820335697429248772856e-24', '-9.455691040938667713520206049214946596676072547630808599e+18858', '-4.0587412586573348e-18883', 17, 1);
  t('-8.9043692e+441', '4.922502215686303689613411049123590850286314e+575', '-1.808911161405853749472468658351477252530084855738408407062362524906222612956452727322923633285164153e-134', 100, 4);
  t('9.983855174767877564750840e-16', '4.442077176133267221847948159903169825858e+4843459', '2.24756454669673458228e-4843475', 21, 5);
  t('9.1600446121e+70', '-8.8825588136102715227205181620111211101906144801558076662e-70392', '-1.03123939896e+70462', 12, 0);
  t('-5.455091346996779943893118119386347518740324439893518e-378', '8.45021012932221140260940220693e+149', '-6.4555688716753023998259305299997301767155121789813739345022368676343759e-528', 71, 6);
  t('4.92731241140976103504385528e-85', '-7.519758995215778653646e-775817', '-6.5524871402722027658916864184976681654878155377984377065047487e+775731', 62, 6);
  t('-8.923714924347837557972462364680155791045288887882610e+6873', '9.56049787920712725220195633365229138986759e+45', '-9.333943730855050949857016361651494295053372418814941087257474684725684596833663177979308872e+6827', 91, 0);
  t('9.120994782870624984584405564109022983116068171157228e+22644', '-3.928918196021315074729587563484036421484733019e+3670', '-2.32150284831767517357787307199788401065025409196970414726140444186928253286397518539667412e+18974', 90, 4);
  t('5.260714966994821283e+7974', '2.2913553636054275189474093725394183782817319448922e-9815', '2.2958965905302146301937347743405869476376213710869775730184386683339649366593429618e+17789', 83, 3);
  t('-4.4740423949e-72435', '-1.99011935060167359745521083763866204854294900538e+336309', '2.2481276781452132166604677212513978692998883937953944582350859781573893143723276447e-408744', 83, 0);
  t('5.92364961196923e+621', '-3.07947537997234899379967457947260721873267015441e+64', '-1.92359050846589954327755584024104053274424702849524072478922488187e+557', 67, 3);
  t('7.4307644562561968240753606201216532907e+101310', '-3.054126964798505532855206526040009849562e-582', '-2.43302408246359126563744935804440189495732397304842343038283559694e+101892', 66, 6);
  t('-3.6515736e-67', '4.031148294860998414946968992580399564234775910e+722', '-9.0583956056767024362785126665230736574031747156e-790', 47, 1);
  t('5.12232095778e+2370495', '-6.760344688038305e-44250', '-7.57701151960992474378716794694436925607132532879604376221126604539e+2414744', 66, 3);
  t('-2.6171943577487375211769996917725770478183344962847645e+962716', '5.73323562459228e-723128', '-4.5649516767154668110122243660466465821745959913049747153161639490316092433928472494475807769621e+1685843', 95, 2);
  t('8.821270839203e-679', '5.07429888713154541185707012184700541507e-9249', '1.7e+8570', 2, 1);
  t('2.29412204114808702060785497978943701643961929542569413e-6510', '-9.622814515785763564752210878e+32', '-2.3840447484307e-6543', 14, 0);
  t('122852209.33596955', '70.15723957573339050956281378262617021489887806918', '1.7510981058e+6', 11, 1);
  t('-75049944334321968158467581412161257581486904.997273845949678', '-4.082759261570570772444092418194591442e+74', '1.83821625342297260377506424790352766382547125503984388883015500358689104592368594475582683259678766e-31', 99, 6);
  t('-4.5117823143987985011645809838922723', '-6.00323433799578982350060902993223313807644382e+4086', '7.515585866509885184268837828897735923269261979645976910830709421410102482169159766563263e-4087', 88, 3);
  t('-7.4593069795167803409689743730979615453017626076532e-14', '5.5663346748483061802960115581505188409329657206216e-7', '-1.340075186858947079885e-7', 23, 1);
  t('-5.98823559467477311715849890335487330180e+11528', '-0.003290162508620091707157720499945878173131108817', '1.820042499112381186000767125637133671860118544803644749917362074e+11531', 64, 5);
  t('-8.26599479873320138912892557999e+28004', '8.900215927286791832913039917332244601e-6874', '-9.2874092789039432437557331596139479758721345609181199154644688864812475e+34877', 71, 0);
  t('-1.02023863943303515301017615619e+4322', '1.10816762232827920665124979716e+93', '-9.20653716e+4228', 10, 0);
  t('-2.86670168042560191292495661545255583', '-6.110204394707469252122e+5003425', '4.69166249644394664609668567108779207194760268516395373342e-5003426', 57, 5);
  t('-3.2754199252801842e+61139', '734417820.90652525258577181729193558978088046760353', '-4.45988622830146565668192617213406602125787940156794498910758882e+61130', 63, 2);
  t('-4.8114404362823181773e-460582', '7.076937099607761048243002399544008585999255767779024301e+4204', '-6.79876105801334884e-464787', 18, 3);
  t('-6.55206394837274730440426799270292389479407549e+19106', '-2.8756448116862e+42', '2.278467744606746127818328395006356350085655193447054475691515164242056420749349939129807249261e+19064', 94, 1);
  t('-3.74215532185692341306475613849903028366059824562097819775e-36588', '-7.799292840730172469140913e-91', '4.799e-36498', 4, 2);
  t('7.403316510551448450015823234611e+84', '9.427386319491878362336646183983738886083245526967593306113e-959113', '7.8529894285168912383267263952350231825317953199580227385482034e+959196', 62, 2);
  t('-6.495564848717913850474651218168419900484433726760148327e+9447', '-8.28504059637611534949554557191770292978508196249942660685e+23213', '7.8401123e-13767', 8, 4);
  t('-2.47788321661263918678937483451126081321445035903756456684e-83587', '0.0000211725639826162452333210075163338973686809659349232', '-1.17032741931827797153160037796004944997696446802029767333159344823471506e-83582', 72, 4);
  t('9.695e-11', '8.40163469685e-2967682', '1.15394210172395588925455912183e+2967671', 30, 6);
  t('-132.33407303466944376', '1.466164851817703355310499908522445029294180477755498710e-9277', '-1e+9279', 1, 0);
  t('8.64e+4', '960.48225415038070623990188582991488335939999902826672627897', '8.99548113738211054297392811171307844646e+1', 39, 1);
  t('5.76902324039256465315903434891205015e-14', '-2.06583234799685176901e-8', '-2.7925902341429288008595625270227784057263759253579132018454586176243668723938326054166325985390393e-6', 98, 2);
  t('-8.9822947743878120795004898246535294114e+40578', '-7325362666.571509696', '1.226191136635122599643895425505792565835728664039384330091318557180407984887976e+40569', 79, 2);
  t('7.411696574592452492944841579170107451491527089716e+302976', '4.60e-64', '1.611238385780967933248878604167414663367723280373043478260869565217391e+303040', 70, 5);
  t('-6.2805e+17', '-8.14519184598728277263411977061042382499210282e+702', '7.7106839455157577783689155766117434339899018121092010961599488908636077855522951995e-686', 83, 6);
  t('-2.8398440377370809032542023717780905984097412510e-27869', '-1.93622897192071928506033735794847981683038501244695388206506e+5016185', '1.466688123626197339169243462412413297918925848556589849020950423396516735218e-5044054', 76, 2);
  t('1.0458363877686671976882334204321e-758448', '19933633.4230586928695660055656', '5.2465918559477054268771147104906644541657992619773953716966e-758456', 59, 0);
  t('-8.961720064943983913809726185290897467761236027926e-34', '-8.54549052659300653125164e-1656624', '1.0487075068488693931384182600836645475478951778840793459989591994005165139e+1656590', 74, 6);
  t('4.30e+906951', '40771588.456183336193414605924790075225845780200889', '1.05465599031569513067326760432739256111800229582599411083097e+906944', 60, 1);
  t('-4.31084416090164708660205e-7129', '4.860738807588668796955241690436591422767347415963e-5582619', '-8.868701511324746031924325113140342087764175889291559606764508669018310708707116574358e+5575489', 85, 4);
  t('-8.29149114e+780586', '0.00008544744301211095214406291719339499873664232', '-9.7036152841048734444845633309028372074266373e+780590', 44, 0);
  t('9.77129327906576217122091545215586069348608483973944671214e-7', '1.8079e+28684', '5.404775307852072665092602163922706285461632191901900941501189225068e-28691', 67, 4);
  t('-8.20793963126552000537116838363086168572853745991121568541006e-98', '9.64641262763382542567710e-51', '-8.508800056668165397129310597666269931308661076836174672419479896e-48', 64, 2);
  t('-3.6176843981400574677912736582474339026e-48453', '3.02183549648586787061e-5539', '-1.1971811180148985936879986164134867347294384304e-42914', 47, 0);
  t('-5.968397680044276695651584e+9096', '-4.614517534700289e+64', '1.293e+9032', 4, 6);
  t('-21', '-3.70850481417723207e-614', '5.662659495470833e+614', 16, 0);
  t('-9.58661405316543228368760763975257686567373339e-7358831', '-2.313910171251627663e+3318', '4.1430363945286144463766335205684117955699333404499322374415e-7362149', 59, 1);
  t('9.65292758714380486538411461404177406107749655e+1440', '6.913227056152443861643199181616e-7', '1.3962983580227062849823277930997746828646307420760691273736897836246753675296903744375626149e+1447', 92, 3);
  t('8.28174704078297432040643187958300524608847208333945719e+62', '-1.6483284829858043084e-818', '-5.02433048161693276182981867844595002e+880', 36, 2);
  t('8.007467117669440167e+153', '4.898987298575040484146486444e-3508', '1.6345147741041413902344373075023182759224927102422822243087e+3661', 59, 3);
  t('-1.7319482835320547453650748664417271887834e+76106', '9.07e+8', '-1.90953504248297105332422807766452832280418963616317530319735392e+76097', 63, 0);
  t('-0.0000071078844445129430615963499634064247374302210295295', '7.5313362575336085260948518769569382108628333091694834619e-8632', '-9.437746770903920413267641328108432493433838e+8625', 43, 5);
  t('-6.1022385943659518760557754527900393e-3645', '3.03803883650519539927744887222990463938825e-9462604', '-2.00861112143834714e+9458959', 18, 6);
  t('-6.9559158998e+448800', '-2.83723925006274622872827466006593836684867e+631', '2.45164939814158014559493e+448169', 24, 1);
  t('-1.86525869429085843675029030328801229562176172716503e+4891', '-2225.25499400768849598889728', '8.38222450601728103256212833080730582971587606623242284309192267e+4887', 63, 4);
  t('-1.906282980853706848014890041641367947435380267110901137e-3574933', '1.393506253039343502489912911533790314655218002047561960911e+5053282', '-1.36797590731721379975696052094740594e-8628215', 36, 1);
  t('-7.942417661436444666155e+243', '7.9066254470361738759260423817051935299974117e-718604', '-1.0045268635323667224715069595144811241329305690943480061913796102858899724016410305300273325e+718847', 92, 4);
  t('-9.4258441909533543e+3824', '7.5311536707110947031690119091143449064693815063e+7969', '-1.25158038238029501255670509975896601916114366217107847634e-4145', 57, 4);
  t('2.67526716565508e-5542754', '598.5082971777964923719713', '4.4698915257650120185982682961161925623265030562587694578125e-5542757', 59, 3);
  t('7.357388541446e+903', '-8.70e-80439', '-8.4567684384436781609195402298850574712643678160919e+81341', 50, 2);
  t('6', '5.069e-636', '1.1836654172420595778260011836654172421e+636', 38, 0);
  t('-4.5006987760521270831425994465786984664029797647500e-2740', '-5.8530498514e-6525', '7.68949332453677635e+3784', 18, 3);
  t('5.610359611286545027530565260019970209914123141279299572e+919', '-7.29792650493826357380686644e+915', '-7.68760771637012473609298163803943493633238634644086242284276111447734152268959520000822891579e+3', 94, 0);
  t('9.00460498005390008542e-158', '-9.8366977177e-2228489', '-9.154093414755599067868670651404132249601089111649809338330929363778511792745276829e+2228330', 82, 3);
  t('-4.31679605994773674e-200', '-6.2903952146452115088e-1380', '6.8625196234052697e+1179', 17, 0);
  t('-3.24019021513028894806e-7925', '-9.896685402e-420154', '3.274015575432443101579758592390992120980021832162105136299047126162250822651743416e+412228', 82, 3);
  t('-8.93283097346249e+596', '-206.523149385868409472', '4.325341251102250290661334596938869666714176248e+594', 47, 1);
  t('1.706099938516851624077006505602294759546662998e-6526', '8.173763490727923844908558949e-8', '2.08728811452912854602428498444772623223702284906999212887528020533162563531780103382e-6519', 84, 1);
  t('-9.1420418890725205738446381441692606915744140036041779681e+2049857', '-7028.51299481467620386544270009618145', '1.300707830492326305594803888902470033688553715e+2049854', 46, 4);
  t('-97844933.5221767675567896378267842185851320254998283854279', '-3.984129207696201e+16493', '2.4558674787245421182755980924171950909759385497886788661291e-16486', 60, 2);
  t('-9.8085e-422015', '2.753411590324229807865787348169528242e+91', '-3.5623079507866071e-422106', 17, 0);
  t('-7.2082319041935610025852720095088945984474220489692e+760057', '-6.5127446850e+764', '1.10678865099616000110178923888045744855533547588610868445244449191854e+759293', 70, 5);
  t('9.4949511094413271181353095e-7', '-1.24729106197516e+6702', '-7.612458229601600865603423417703e-6709', 31, 3);
  t('5.830e+9', '5.3982762998061964186554130446166827650082856620307e+99', '1.0799743614845e-90', 14, 0);
  t('2.86615622875847212162366436370915972265965884e-5476172', '-8.40255e-3962269', '-3.41105524960693137395631607513095396357017672016233167312304002951484965873455082e-1513904', 81, 4);
  t('-3.269001223052524569046431792699991008948195216420357774e-1055', '6.157009595957380488387716880092084e+1410', '-5.30939764199638736845678477540631651720812563006691070214554660430398502501459003e-2466', 81, 2);
  t('2.0576609123471778786976374890818605197262967111915944874206e-7929618', '3.06513502427732802e+61576', '6.713116701383541863881278061804986519025581619685243593454921929406481357691445292070279e-7991195', 88, 2);
  t('-6.85006782303648011e+174', '-2.7778230253422225241763545329403653828444e+56', '2.465984247571914613320915578953410428134545546025707559e+118', 58, 3);
  t('-9.52136e-5293', '5.59831195288791076586059401808341296063420272874517134177e-29', '-1.70075552776017953077524755677982633471486048609444011953018554104509035575938786360609389e-5264', 92, 0);
  t('-3e+60', '-53399419.8569659', '5.6180385630325402341196339417374033872626693292561486830335778243264266518716597e+52', 80, 4);
  t('9.39182367043563579687809217392347e+6340', '-4.250798904654189760777762103e-30', '-2.20942e+6370', 6, 2);
  t('5.231864e-4086', '-3.28882441273043192689015188108526433185e+69', '-1.5908006458929277793631053189e-4155', 29, 6);
  t('7.940389593197326135731489e-811', '-0.00003654997953972994164737553239538850432513732114022407000', '-2.1724e-806', 5, 1);
  t('-4.3228e-1116', '8.864865600e+73676', '-4.87632886391419177296946272936162732123090506865665284310683740089641e-74793', 70, 1);
  t('-7.4069871103057949618107450715592133674212655227878996e-29192', '6.8e+70', '-1.089262810339087494383933098758707848150186e-29262', 43, 4);
  t('4.0625650056253826300522320976800667101600873578330069374116e-192152', '-2.187469292246038738933083932611541230120e+823', '-1.8571986450397402047180017019e-192975', 29, 4);
  t('0.398403625460754165392379235799719286513099098812226095', '1.928598171255202747846347676206686401776333632651588679809e+245', '2.06576793133355719719522812910771e-246', 33, 3);
  t('8.45165895812645331905668372378472054458248243354e+7729', '-1.0808170932433517e-9', '-7.819694017573718633260261936900325520474069230490279124e+7738', 55, 0);
  t('0.04038302045383689737537715305', '-1.32939446599768226497472436070147847456299779152729975e+7854', '-3.03770035807470430831018e-7856', 24, 2);
  t('-9.13633591654216316034e+614', '-8.00626', '1.14114904044362325984167389018093341959916365444040038669740927724055926237718984894320194447844562e+614', 99, 4);
  t('8e+28', '8.703838751073686338701653179112604871e-47614', '9.1913467480232646270448834541777807310648243266524730865928427e+47641', 62, 0);
  t('-6.434512167015392179622e-5531245', '7.408522842627232539e+6763987', '-8.6852835628614546067092282202348002625881186498390751169432899715675788304629e-12295233', 77, 1);
  t('6.0902487774260370428642976805816787645545214e-6148964', '7.5854119021155769446167801709686445998651364641091e+4694042', '8.0288965925e-10843007', 11, 1);
  t('0.831725477620089540637251608898947983', '-7.46450893627041736511113807836155914e-8335', '-1.114240045421735e+8334', 16, 5);
  t('-4.09008155574639725039819291788738e+13271', '8.9971609112347516886481809085842319543', '-4.5e+13270', 2, 6);
  t('-3.2120904155341355747476839359766307e-9', '-7.8836687595691879463347598866117461e+1176', '4.07435993760557732e-1186', 18, 1);
  t('-5.1752466278763480562829198898656891176023042534294070243e-8246', '1.9377683998247856651135e+326222', '-2.67072506e-334468', 9, 6);
  t('-466866.110860556329593288994009716567594004453227', '-8.853628717569284701e+75', '5.2731611608480896934533923886716855438675844611536944119677243703e-71', 65, 2);
  t('2.3265655965468872571645946973229428895228525e-5966882', '-6.82123278223710542769086195351367924671468484078e-795', '-3.41076997490160723531099556292e-5966088', 30, 5);
  t('-6.515253415e-1689939', '-4.9409375880627692779120032067933438865277266139028064942566e-698062', '1.31862694e-991877', 10, 3);
  t('-305865320.7641477', '7.22301542136e-77', '-4.23459321240874815010765995510689114118693492253208681442304908334041092e+84', 72, 2);
  t('7.50585975540953514e-58849', '6.529777913039545664460982387159988245322879e+895268', '1.149481629e-954117', 10, 5);
  t('6.473006748046229360908291331313309523793e-5755841', '8.02336037096096949319672589791582590501057e-964042', '8.0677e-4791800', 6, 3);
  t('9.54840299023736708149984236017234055477610043557437948e-272', '1e+705', '9.54840299023736708149984236017234055477610043557437948e-977', 88, 1);
  t('2.3026675461e-906200', '8.43757563441351707e-93124', '2.7290629985091148594510005022879253829661e-813077', 41, 3);
  t('-95675736706055295.677406551742811040917583753585191172113196', '-1.820098938888938311178e-733', '5.25662284955012491159526249792727745494970648252567199389049386376e+749', 66, 2);
  t('-0.00000367878882129254076004368914186463', '2.21534e+7414', '-1.6605978410955161555534090215789133947836449484e-7420', 47, 2);
  t('-8.523221035207e-290', '-9.22950311628931292976250786203698455520043289656e+86478', '9.2347561161382748540585461990341573507302574000703952620656129161e-86769', 65, 4);
  t('6.1e-558', '5.083820386089577292843972361554693918170842e-8020', '1.199885034626893592002313284596062998690006879533206435358120004976236854862833e+7462', 79, 0);
  t('5.75437017863019687e+5601793', '-50265686011064078917406015064.05038', '-1.14479093697509492717111975944408926832484246759405446142254875451648396850802376531153446864527813e+5601765', 99, 3);
  t('-7.02479085615239757380511080902628306127526377163034580709690e-1929109', '5.937151585632715191e+2105098', '-1.18319210059444253820374823475694931242101809142953904771552228077106021985746268e-4034207', 81, 3);
  t('983326.3529671', '4.94670376809815834641010722044433880038e+1258390', '1.9878415993063518243178667667199833409018125874458e-1258385', 50, 4);
  t('-4.75992158e-5024', '-5.74257395010042921477085748004249382743005431374588507872780e-74735', '8.28882940186909728377127651680904728016558720061e+69710', 48, 2);
  t('3.602616015331472560955592560521969863314446783038758114320e-51892', '-0.000009441970853418132', '-3.81553392957919659196739917915e-51887', 30, 1);
  t('9.1497455227169232552314436076169103347782270651e-219', '8.373249799378905593133718575799e+456533', '1.0927352870083e-456752', 14, 3);
  t('8.4683622821934687328543351512700150155587470586503736e-78', '5.91343666713537280212039965446622473725e+65', '1.43205427890306135113135547671752046828696421139270728527469421798846833607098329784358305203559283e-143', 99, 6);
  t('3.189138486227352332426732440e-598458', '-621882029.123105326595384964817058122413466827221502', '-5.1282049277485119126158079163601873401925833903091364654e-598467', 56, 4);
  t('8.36327002464142378308e-5456846', '-4.1226702772597622341370819150656144937577e+170', '-2.0286051180887267413705307718807271002754666407162366416533614250649191231300041619677e-5457016', 86, 1);
  t('-8.374182824415045331433178480296e-9', '6.147680930636187947068915894326103429660857246967961753e-897', '-1.36216939670427065891719674e+888', 28, 0);
  t('-6.1555075166107317522046137450664e+1059243', '-0.00000787', '7.8214835026819971438432194981783989834815756035578144853875476493e+1059248', 65, 4);
  t('8.878418479987526159e-22', '-4.64013766942433523967e+99058', '-1.913e-99080', 4, 4);
  t('-1.643833156167073156227836724712230496026e-31390', '1.95796758465874581171150957275752554e-5636', '-8.39560965690643367874396342e-25755', 27, 0);
  t('-9.109741755526673003743735919690524222898437842e+183', '8.236165559556987020556098725141774367711141832664077433e-9827', '-1.10606588583640304424494579923987e+10010', 33, 6);
  t('-5.4149584e+8284753', '1.54056e-560104', '-3.514928597393155735576673417458586488030326634470583e+8844857', 52, 3);
  t('2.690472873319424346621055802539580796027419185779186808860e-24432', '3.268504216190878954310096433055532399962763307309182634339e-72997', '8.231511099150138356502834964202409620452717229808497e+48564', 52, 1);
  t('5.511122960487624576329342700193361e-4398', '8.2470250190446060067026043384322816060e-86002', '6.6825587987922367549111897380349609664e+81603', 39, 4);
  t('-7.917e-98', '4.79652886e+27883', '-1.650568615571719920809566441345252324823914433822462188e-27981', 55, 3);
  t('7.428249642380234695622316151679325106710e-62390', '2.4e-977', '3.09510401765843112317596e-61413', 24, 1);
  t('604.4744600260950239251801774458251452270154165496606', '3.85136383475361979e-428203', '1.56950754579842125925300989434603040384295630928408984826598160297536279981120155879629e+428205', 87, 3);
  t('6.67e-284', '-4.72088315743215515882516e-40', '-1.412871231413410813e-244', 19, 3);
  t('-155151.809518011593063914917713333906645884471601871177', '6.338188022666799058422649809028e+39', '-2.4478890333191995048e-35', 20, 5);
  t('-2.22820726393277647144506439090068833253219972870532e-46', '5.56841444938689968910723598834822695961194434430534984855e+68617', '-4.00151117375630911580611713636285050732371989524994875555666134152593754919e-68664', 75, 1);
  t('2.338716212847876184951897169726620037171397775970546724e-9', '6.7256468684977240454503848810066203901786164055e+491', '3.47731044845990282013489394411795073677106e-501', 42, 3);
  t('-8.230327025182274219825664526246374e-668', '-5.0930940348440986089411798148973918058986185512e-19276', '1.615977825831406902502437049283858159926001981843491735777952747247195e+18608', 70, 4);
  t('2.3392164138920437971579e+704', '7.31e+13', '3.200022454024683717042271e+690', 25, 0);
  t('-2.79932318853820678394061e-21', '-9.65201e+795', '2.90024895181232384129379269188490273010492115113846753163330746652769733972509352974147353e-817', 90, 3);
  t('5.262694813393922563321996504321750944298404865e+575', '-5.27194871272972704640595467498038359604198471206982404550e-6428575', '-9.98244690940664904824180323510480234891254747217390046867982787938145121410706590862433793592286912e+6429149', 99, 1);
  t('-4.3032495194572e-1791', '4.170871137116609104632427816969e-8', '-1.03173878501365214057079824e-1783', 27, 2);
  t('-9e+8', '3.2664588981099192225e+355657', '-2.755277e-355649', 7, 1);
  t('7.9856701729158152160178915270177478260463473296e-98', '-2.15e+219', '-3.7142651967e-317', 11, 5);
  t('-9.24426980659693e+5552', '-2.360371109184588582050804305650031929948074e-2907835', '3.91644761e+2913387', 9, 3);
  t('738078.09969552153351935869902827601706320582953', '8.906309895128553028387337269294725851683085877664e-982516', '8.28713696678379691594990928450730337629e+982520', 39, 4);
  t('9.3107363647475354115363146495992673271e+5571', '4.1390180629e-470', '2.249503680161273502408593378453233066971837302343559192685348743e+6041', 64, 0);
  t('-4.6488388360919e-9351', '1e-3552626', '-4.6488388360919e+3543275', 37, 4);
  t('3.502125971589041207828696400973199183170931575e-82', '9.904207204795497308966962641344218578493956925920528420e+3548198', '3.53599828756950296897419759303716701189355473124227765928807732432119654387837196648e-3548281', 84, 6);
  t('9.626778775512443898886198120e-9711', '7.65144880552399643302293049643e-419330', '1.25816417520330914099639287218543216722553820660528264158268029998306536564450700364e+409619', 84, 4);
  t('3.6373670222495719639910226014743e-9583639', '-6.07946609592801701149048389895673e-7772448', '-5.9830369391908516894138809710480605955155889487697065431530475e-1811192', 62, 4);
  t('-5.413414359362491050376993e+974936', '-9.24500668218936221853305801534354e-1217', '5.855500753494869122525762250264453613312731841180143e+976152', 52, 5);
  t('-3.69528652339681914004592099858729090908267654e-539', '-9.2994862604892540e-1857549', '3.973645876651262374221097377505640866928128627967019046518019594709834308762109480776941e+1857009', 88, 3);
  t('-5.603651212122744684360301186677920904987516640e+797071', '4.9958064506424300553261469051348812542e+10837', '-1.12167099896397097658540051011786267488651144897649339081833e+786234', 60, 1);
  t('3.72422571684e-5479', '6e+261', '6.2070428614e-5741', 35, 6);
  t('-3.16039807859960646663838964073463294295045516614321232e-304', '6.299543644013528164775335365232981750549918232246358e+885967', '-5.01686829585337e-886272', 15, 0);
  t('9.67423623391707022179457264639757180688635006848e-9402', '495973672.925098457189712686141111899121970990995137558614709', '1.9505543866595648012867e-9410', 23, 0);
  t('-0.0008581694', '7.820967675296252e+74422', '-1.0972675449237082155925496257359580492418377059710845942002508336411181101106074e-74426', 80, 5);
  t('1.226198817917201442232008153782416893056584e-1854', '7.189408017858061042470833e-25', '1.705562982197416890856662477715422791996245047064069983970266716961215470414672e-1830', 79, 5);
  t('-6.35189227314088953150171322959882305313405e+48', '-6.694886542643052998499564284e+473123', '9.487677248423161703764467275166572204266404817544554045922423528094758875270330077509010918430467e-473076', 97, 0);
  t('7.4616536852330028276844750455006760e-8213273', '-8.2616947301811015785381280075461055528562843038319069159e-89', '-9.032e-8213185', 4, 5);
  t('-7.2684620847414561029923060748011961389878918997817e+87108', '1.948949090678718996919e-536318', '-3.72942634546202e+623426', 15, 3);
  t('-5.017576430311015662375160024056601527714e+8326', '3.4813441259481e-48', '-1.441275624811879878e+8374', 19, 6);
  t('455317007.49227692213377339503163384422094806', '-5.65275506053420792162316654536951391432754e-116', '-8.05478041444179693663347923332141134756382641597760190726596537031004841221031083469960305598910834e+123', 99, 2);
  t('-9.633607066849e+57758', '9.02343831685532970072242113378955028698e-515', '-1.06762042677832747096251659449918592253561599947896316645079438705808961282465257e+58273', 81, 4);
  t('-5.53047387401759485491421040802881710250733e-60', '4.832438868686975373399225473898e+6688', '-1.144447767327946138566275743e-6748', 28, 6);
  t('-8.5517e+999376', '9.6709761353725e-467008', '-8.8426440933106609382861204631117022322680864436317976499429295071253175787715682e+1466383', 80, 1);
  t('-4.7985931982836001286789522e+3083', '5.8754920117e+84', '-8.16713423952930761956558239736905197909929787063589141361439526438153186264845177340265534379e+2998', 93, 0);
  t('-0.07439743626604329095911613823765871837095179096615', '-9.567422335725e+374', '7.77612126395229348289773448812196298503460679537357436759941658658788819681845695978744714880971133e-377', 99, 2);
  t('-9.3513764e+35', '6.3337512065852698115383106285012045646880436155304e+9430729', '-1.4764357005809247057710477075102868262504333e-9430694', 44, 6);
  t('-6.4526743163882376690571082e-523', '2.67548292599e-9608', '-2.411779291770503887706294874287328174391375e+9085', 43, 3);
  t('-1.7351679399705858734112711e-7', '-7.700225783512425379281739340851003461459979259249134278e-990', '2.25339878174208077097179e+982', 24, 1);
  t('-0.0004775444877931838264447996', '-9.45257809685942648262068211096105377974029e-9558', '5.052002563743384263167828462454906927095954922595940600820474e+9553', 61, 5);
  t('5.61325106751824e+33478', '7.0e+730198', '8.0189300964546285714285714285714285714285714285714285e-696721', 53, 3);
  t('4.54882500559155710013485527076437069018865104e+9974066', '4.8906436370421086920470075e+952447', '9.301076388265971471355717622194478e+9021618', 34, 0);
  t('2.93236783716e+32', '-5.663565141e-50', '-5.1776006175541929729523349363379380260226091394399307395553446853e+81', 65, 5);
  t('-7.30289545768349143493882e-46', '-5.144108787179330100997948996487019772996e+79', '1.4196619394761845879192934848753033761430984e-125', 44, 0);
  t('-2.472060745e-6363', '-3.7231031996015984083496038307e-2183', '6.6397857176361109727293604341430323063265900296318742853356936680116395513e-4181', 74, 3);
  t('2.47103852713966506e+686152', '-8.526641704891077560618072669077e-574980', '-2.898020830079233415870600138873590163204027879e+1261131', 46, 3);
  t('-9.25e-89', '-1.5070207e+9915', '6.137938251279494e-10004', 16, 1);
  t('1.119365099713806799416359', '-0.00010141064', '-1.103794532520262961969630602863762618991458884393195822450188658704845960936643334466679236e+4', 91, 6);
  t('0.0070606848678199435435749361053874132790999688086830', '2.814984075612565257476863631e+880049', '2.5e-880052', 2, 4);
  t('-5.00908986898781781403504e+571', '9.594306468292974628815546812025040297907943190487345e-30529', '-5.22089833751061988567640162835383536040375896870047458122903418260221e+31099', 69, 1);
  t('6.536235082892238482944e+136255', '-1.50376540589127238953100e+66', '-4.34657896589811005907720591974213630955054958458773945203e+136189', 57, 3);
  t('-2.59747179123742e+16947', '2.16814390894744105231364264491757e+374', '-1.1980163219416569074208688960981e+16573', 32, 2);
  t('-7016364441823846934204633960.3246656336726794986082', '6.267890414729140882953463043550540194e+71', '-1.1194140257040614563483e-44', 23, 0);
  t('3.81096696335100355559824657758339e-27', '-2.66e+499444', '-1.432694347124437426916634051723078e-499471', 34, 2);
  t('-5.3001546173537375421251306682432058841675552963350451100e+458406', '2.543e-80', '-2.084213376859511420418848080315849738170489695766828592213920566260322453794730633110499e+458486', 88, 2);
  t('4.9725550366e+208', '4.320790745427404224419192661705787428967725606692482e-3161', '1.150843752815926872292918e+3369', 25, 2);
  t('3.704010e+554', '2.885131087270377727451557613008319e+39', '1.2838272813123245436918555271613317085199348008316522416757736e+515', 62, 3);
  t('1.6173538174534536423307034636137041553e+890', '78043.749364184251171298739776580568', '2e+885', 1, 3);
  t('7785.38926714803', '5.7128906668117937069399910232699277065548249552272e+415034', '1.362775820719992957199284214478469761555796732349660451706183955300180161313835764446e-415031', 85, 4);
  t('-5.07868461604023e+126062', '8.106710194968426653e-9549', '-6.26479113462376589e+135610', 19, 3);
  t('9.3112e+31902', '5.802920972045397225171810479e+9962', '1.60457122281264059215138442381891442719754901809875966906068353237e+21940', 66, 5);
  t('-3.62116210365102980919563849163417520549036635149e-859891', '4e+279430', '-9.05290525912757452e-1139322', 18, 1);
  t('-29741.72386912651648227429513933355112481075034929319615168', '-1.81254559267e+865246', '1.6408814e-865242', 8, 4);
  t('5.8020204908101208375565271248827e-8422', '-8.21e-574867', '-7.06701643216823488131123888536260657734470158343483556638246041412911084043848964677222898e+566444', 90, 2);
  t('4.2249210117868151825850569355007155971e+4443684', '-2.296799414051960498455283054114e+6935', '-1.83948192686679028592157851487994594041898951624008242379071453644286557491435400935466725337e+4436749', 93, 1);
  t('-5.5683967199378466381245121771765180601322725857e-5782', '7.63989281e+2878', '-7.28857964165317476244082562877805357603213884e-8661', 45, 5);
  t('4.890056674046618952861254179424762170739416e-981373', '-4.788878637890516129744512240101667588921071707e-41173', '-1.0211277093880732348828801954836327803781607041506663717693804718e-940200', 65, 3);
  t('-7.5489699907035199753435159498025523e+7333222', '-510.03729222053270862928676321455378034348461', '1.48008196770040397791935767373126144302782359243310019996660175167506e+7333220', 69, 2);
  t('-7.3270584615286236633654430861733084802309157028492e+351', '-8.84687011420092814735489917e-10', '8.282091142908592656216924750530931007470543029276830203292965428595765458179031126780952065735072164e+360', 100, 5);
  t('6.083061611465e+8023446', '6.08718592e+880', '9.993224606e+8022565', 10, 0);
  t('0.0015756454197115404268201989899732397545221953', '-2.2625460695220128276222100052234505e+921527', '-6.964036847410636172915824361955334e-921531', 34, 6);
  t('3.778140914519654343155865295708356480390693e-4233', '-6.36608e+3186', '-5.934799616906564704112837563631554238072240688147e-7420', 49, 1);
  t('-1.037059766939040576436457737042171551398340625385e-2015', '9.58872e+581668', '-1.08154140170850809746916975054248278331032778659195387914132438949098524099149834388e-583684', 84, 2);
  t('-0.0000530084462581851788143636634433353360733168428056102', '-9.612800144197474955e+9901451', '5.5143605883e-9901457', 11, 5);
  t('-7.5993961461969e+9399', '446.47891237020844546821761352121943801', '-1.70207280470520469485545830180759296578670882013617711034068981979e+9397', 66, 0);
  t('6.915626408396995959757568961988119758281756464113147402772e+286302', '-3162.5608939936388120318745', '-2.186717233344410689191305768633537907690605443288196266993964591605791604315e+286299', 76, 4);
  t('-5.5437383828828567108517373e-776', '-2.1140533665e+309', '2.622326602880891233571296555062627365324838406817011636683297512205919991849931381569028144021e-1085', 94, 5);
  t('-5.613748719378402450137898026316048568184578645980579e-9740', '8.6217788896525220020177465853e-64264', '-6.51112582591949275423601496056334250886183469397e+54523', 49, 6);
  t('-7.23012652470383234547804544659544027917108007437e+92575', '7.9142419737684090279744728848831732235974822361464930703e-1068', '-9.13558942052560035722583534016743762565074490431624107788163e+93642', 60, 3);
  t('6.51068003e+842', '-6.64386076662392515945318968640865e+889537', '-9.79954316e-888696', 9, 0);
  t('-9.4220842814894740355643871550053e+81204', '-351.6319547964917916245896910470732083574798711796107577637', '2.679530160148e+81202', 13, 2);
  t('-3.0704509983896666335891441257686892358819812695e-87', '-2.92831001734855596829254121942366818e+3082', '1.048540277565902156270373e-3169', 25, 2);
  t('6.926278921765102227557189450', '-7.812398429709661207988000287030472469e-5915', '-8.865752283e+5914', 10, 1);
  t('9.53634e+9', '1.965138983505348446115414087092375e+2823774', '4.852756003541998481759701658454e-2823765', 32, 0);
  t('4.082089070477382148438037008743948177675540493962936e-250', '-3.70981132278431827653065681305312802364964588974562536682743e-150415', '-1.1e+150165', 3, 6);
  t('3.21546866478602997320941159e-7809804', '2.62951330458537889745968954585764480040058726479e-950855', '1.22283795224722941388512706360049152523621791432774199557610762019505424325062326677e-6858949', 84, 0);
  t('0.000001127403718694283526088035899517087999169464688', '5040.5820754022433780182', '2.2366538265410857475691712665224191379092605641551473288691761417416e-10', 68, 2);
  t('6.413972006320308603410995781591590851e-444', '-3.2406899986995856418987390184575532612629808e+462275', '-1.979199494210827954302790460021418282864705538491682126209e-462719', 58, 5);
  t('-5.2367600337694e+824', '4.7e+5112625', '-1.114204262504127659574468085106382978723404255319148936170212765957446808510638297872340425532e-5111801', 95, 3);
  t('7.15018163450037e+1043420', '-7.696127791261132765e+2500843', '-9.2906222823110103051626310097830063941749614706048755651973922142971166784633139143550974e-1457424', 89, 3);
  t('-7.7870434680448997482202649e-701', '4.890909233287873419915520169498642163367953112705806720e+3297', '-1.59214638763805558495114202287446668688246661401731068853250999357151987686894e-3998', 78, 3);
  t('7.90371913475853302971e-383818', '-4.441915588351562702723111858272999177138721291129e+95233', '-1.779349241909317480964260091333971618643975276275877872610237534154609112143514e-479051', 79, 2);
  t('5.52588573333732102332213278851769771097086764096e-8724', '2.02383031022516609043346e+50', '2.7304096126134830768336584365822907780869250235184735679058135774495742756972e-8774', 77, 1);
  t('-7.951110327860274247709e-925', '7.2814109864640187205483337377e+33027', '-1.091973841696507960691114551726008615002680220148021354595019556540182019932304e-33952', 79, 2);
  t('-8.873697338599954480139605635456864714416268787583e-738803', '-5.91062688993286e+924', '1.50131238256873842251701844999129311191366440778955739978014259056389850612452719811e-739727', 84, 5);
  t('-9.1436152843889238774692104695050201272912884890200195506754e-1683102', '-5.1803330099891957216255347730391e-164215', '1.765063224074082085292551095234045505e-1518887', 37, 2);
  t('-7.29640957067304272880720e+5081', '-1.33628370095834306203522543771616626623455384353366e-93', '5.4602249248720717426278865744444652435574141226720132536e+5174', 56, 2);
  t('2.33133930445796588877032528e+7156', '-9.9677887295e-61', '-2.3388731119e+7216', 11, 4);
  t('-0.00025785307906304864870180865683793673660493', '-3.0100e-2761', '8.566547477177695970159756041127466332389700996677740863787376e+2756', 61, 0);
  t('-9.6770193467199039717825616309703958882916034534672395359137e+17688', '4.18979882378258924597780062e+496', '-2.30966205150236810986706968918539917699847241253752508255359e+17192', 60, 1);
  t('1.86580566344553757970697514420e+6799650', '7.183157009290574082361044470e+8288', '2.5974730345339e+6791361', 14, 4);
  t('32247647.57900330176', '4.95060627882350112486901533975e-743', '6.513878455037808507585850262360816147927774047594341235776473855525437e+749', 70, 4);
  t('8.71e-52', '7.03973305194871690612804091405988801409802e+4222', '1.237262824559650740798598369147166862397725876647363701770872319086241898425822176503588167155e-4274', 94, 4);
  t('7.7259064897661884349401371280937076e-9161', '-3.8203883860072154274824008666988387842750011e-7660594', '-2.1e+7651433', 2, 3);
  t('-5.9e-23258', '8.5803408660842006874930413872203e-54484', '-6.87618369955572119402196294175433788610487746425290374504322286841107516242783695487400261964372e+31225', 96, 3);
  t('6.90798422701265445098778106e-24', '4.0360722484987e+722897', '1.7115610924909040265927784803960678188103834405789e-722921', 51, 5);
  t('-1.6717449184931518514025394010707104237e+51', '-2.6045e-900585', '6.418678896115000389335916302824766456901516605874448070646957189479746592436168170474179305049e+900635', 94, 6);
  t('-8.08874022564473140199035422086674705e-18474', '7e+60396', '-1.155534317949247343141479174409535292857142857142857142857142857142857142857142857142857142857143e-78870', 97, 6);
  t('-3.99756e-8744832', '-1.129777112335753948998878000850812586076825723004920940396e+82563', '4e-8827395', 1, 2);
  t('-2.1986880829005591347235935044e-66', '-5.53128913507041640704954869060137865e+22329', '3.9750011782245561542585386936073468939246936098645050823063861338176931635325497215244e-22396', 88, 4);
  t('-0.000226930118', '-5822.7730802712548122145', '3.8972859644640046e-8', 17, 2);
  t('3.5358228603437829467542565252394409759941451716564387844e+5819', '-1.1211285523789752278505749782883138380764249835122752e+654', '-3.153806807293378336299434516590902750456037343818220344010026e+5165', 61, 0);
  t('1.540945741255801235146421640530441629341e-59', '-0.28959050868264919998', '-5.321119632910562022779265944683536368088159036094e-59', 49, 1);
  t('5.67465911290e+60', '1.82139318766325427065720e-489', '3.1155596448564028586441404475e+549', 29, 3);
  t('-0.000073324629052045985374799060767559177945662087945380531962505', '-8.6235020738126851431568817200653038002538e-45105', '8.502882984711473461151286399869019847200978176981996060734467871714834324909852815521082684024201e+45099', 98, 0);
  t('2.33655464176415788042730153765978361685159812387632344e+966733', '0.000271032054732387189879751975458037143324874109460138147524913', '8.6209531343856555071879118346685369655541121912109514462e+966736', 56, 1);
  t('-7.3857752173247842104384133616842993670558572216964745352905e+593726', '3.5159864077115843821601402777019512966207959178114659e+86157', '-2.1006267831768699436790676448e+507569', 29, 5);
  t('2.297954795295988905597e+35', '4146.8211032798097836971', '5.541485243910057160840790548754435007075456066243014035815812282364190103557437813616906558929018991e+31', 100, 6);
  t('3.01e+69', '-7.42216850812595777997133787752109191066448160807243548474e+555672', '-4.055418570333702329360565078021946914153372178576123824979e-555604', 59, 4);
  t('-4.412068782529924812219217692595557950834519815e-8', '-5.543105213195272592447854502e+80', '7.95956167677832671851525278405651096717750940224758071101327343e-89', 63, 4);
  t('6.0399317841e+79092', '-36677.6188495002669617', '-1.64676224181393229943713741633116474956001090019712342463957e+79088', 60, 3);
  t('-5.0141298538386961519854674556082694550280506139337568380169e-8654417', '7e-7975299', '-7.163042648340994502836382079440384935754358019905366911452714285714285714285714285714285714286e-679119', 94, 3);
  t('-7.29985014324738862670793202560933158762654831493606e+644533', '-5.742610820531e-7269184', '1.271172707220371224664309552725807599674523039e+7913717', 46, 5);
  t('-2.356096137810041e-8', '8.51666515563839e-30', '-2.7664538815996615880251302469e+21', 29, 2);
  t('4.0e-187474', '5.57545409734318035788872379534690e-187', '7.174303527861673281695766646416424586540746909e-187288', 46, 4);
  t('-8.3408180480517571e+10261', '-4.95082841688375001488467e+93', '1.68473179551267957952978522430716289057086493073281654507483485966953015258527807096102260243e+10168', 93, 5);
  t('9.485789339622840784590481050e-758233', '-7e+726', '-1e-758959', 1, 1);
  t('8.129451975131457857e-74', '8.869163209722283113995855593533903424761637e+93582', '9.165974042e-93657', 10, 5);
  t('115297180.9500040358261890643216883791745981094374599890744', '1.6156778e+59619', '7.1361493578734594128971175021213003715591134220857641959554064554207528e-59612', 71, 1);
  t('6.57e-9104596', '4.59782738301854050988146364667782520337e+9668429', '1.42893576742e-18773025', 12, 5);
  t('3.53298148390840216063756828528528731362938e+37569', '-9.693e+1242', '-3.64487927773486243746783068738810204645556587227896420096977200041266894e+36326', 72, 5);
  t('9.680686421339337e+81140', '-4e+5337866', '-2.420171605335e-5256726', 13, 3);
  t('-3.1049010007243138532000993279548588709755e+1456', '-5.8689398253118613130602e+73', '5.2903950170580031454614795341541860027427127591621462e+1382', 53, 1);
  t('-385306.7431768853361651458', '5.299549444959349552757082715023594875873292038137301e-6847426', '-7.270556623325190052068228882650212113848551007603369953259657763300398903e+6847430', 73, 2);
  t('-5.435380099616010286e-158026', '7.3872183184755192092675802546', '-7.3578170635922066041185497089932822622241189992455124805127369709866132541296736072e-158027', 83, 4);
  t('-9.8161192725730583900346322467838080325819591676675e+67273', '8921123217228.625485161520374979079834199241555733', '-1.100323247818839811644712336352805812090694235673509811515801464524277016494901066683240684988785e+67261', 97, 4);
  t('-1.864801574014925385704410998424068774884363551529322918439e+120', '9.92571328569748e-9', '-1.878758251764155965301353804071955e+128', 34, 1);
  t('2.217008116600306723453013180713380171420e+100835', '1.14660112752e-28', '1.9335478252978045906788601940388402134544588573422e+100863', 50, 2);
  t('6.20e+34', '-6.426989860828637290655138377524103575311848105114906759462e-1672', '-9.646817770458764510831701666413893777968947562533879503975426e+1705', 61, 2);
  t('9.727031936826544042916483357e-85', '-1.89632939742514066776200722457594141263447e+9547', '-5.129399960805347109409011815190515409432537267368130140053722143e-9632', 64, 5);
  t('-8.83141803151495457159535149e-251704', '-4.844900529997879600e+9506665', '1.8228275228426247370397069826926702651387770675304659480581723315579117882554000331882e-9758369', 86, 3);
  t('-2.7746632190032869089126814262857001229975904e+74', '-3e-3595', '9.248877396677623029708938087619000409991968e+3668', 86, 1);
  t('-4.8913969468539595739229875634648105516559842059174585002e-7160', '-4.0157e+5', '1.218068318563e-7165', 13, 5);
  t('-7.8358544722217023164122950825167760865007006130021e+5126', '8.5634284165424837389049497735142038323e+38885', '-9.150370729e-33760', 11, 3);
  t('-3.339303256823800935989097276894281516540e+4548', '-2.6988482496225472744646183693876133e+407', '1.2373067871788737121967452402266068399668e+4141', 43, 1);
  t('-548800521893.8564475915153910333667398576296634307', '-2.5e-9087', '2.1952020875754257903660615641334669594305186537228e+9098', 62, 6);
  t('-3.23665793569776385550085525852436847903310590312e-156', '-9.79026643429840e+8372', '3.3059957636686243240936332506488908316402035527114606879e-8529', 56, 5);
  t('-9.143e+3159', '9.63305968857275e-8821327', '-9.4912730696000098685765139367599e+8824485', 32, 1);
  t('-1.3447e+993', '-5.7289053674945975212302570073', '2.3472197806403512398449179483606256965048752905323068e+992', 53, 2);
  t('4.2e+4697', '6.67283818842789008830786329896620947816689867e-18309', '6.294173305871085754601221156061039486728147858539713818065820568e+23005', 65, 3);
  t('9.69383851251244836506017776859981387098944e-1119', '-3.82486379521501469397596288e+9198', '-2.534427114669977233684743545593026338923169057834944305720357823361e-10317', 67, 1);
  t('3.5355491459875958e+83773', '-526916578.223339198631317192353972585023365789985', '-6.70988405395932650363426244693308215766147991508e+83764', 48, 1);
  t('2.718081113664408632444613291904096612330859388072304159155e+594', '-3.007577482890046655563515677964208993e+306799', '-9.03744335475123100808781769700866206e-306206', 36, 6);
  t('8.8280996164510722828588026346315877842e+49752', '-2.578833566225900100617160782290e-462', '-3.42329172850457241836122644670733151462648422285155e+50214', 51, 6);
  t('6.18681765136134904e-7', '3.732459269136594316351364685108041682930269e-3789940', '1.6575713772738652251097576250282215785789005e+3789933', 44, 3);
  t('-8.323397905126013e+48228', '7.36322886312e+679', '-1.13040054300297319643962874557566832355716223527807796889969229301300843252608254e+47549', 81, 4);
  t('6.9873812870178255895e+27249', '-7.07149107988029991181662887992194025064562374268e+95', '-9.881057909976324172662205032137680894515657641325e+27153', 49, 0);
  t('2.1152163290656455693673204267197398831e-61', '-6.456271859817401712122612921625581674711229e+1338571', '-3.276219426617312200927697655913806547948256572e-1338633', 46, 6);
  t('-1.09183e+7', '9.954814761688306975626749165618904112910970292700419135987e-84', '-1.0967858530145355305021090386646e+90', 32, 1);
  t('-5487828.385013699423427876683829029974006362', '9.70775568463159251831771571625e-3286', '-5.65303512293939798174704195640977462372739276120496777814221708332632e+3291', 70, 1);
  t('-8.098437997145103e-7652166', '-8.36676608646337215744959966866e-20', '9.679292947184936412297378769435397836153709746294399932731e-7652147', 58, 5);
  t('-4.181701311108074144453210348e+84615', '-2.59250909222674097767953708202021830271e-97', '1.61299388443662296483276372047788464624876472791715240051235668441708534365931838065838e+84712', 87, 2);
  t('-2.1946536e-9244', '-8.491803457524430e-4644213', '2.58443758263782946131100536322795270476575807105843776888003208538e+4634968', 66, 6);
  t('-6.23973769513358772836843552665264653907865499141648514068563e+87532', '3.5819515125833579189611124225908919919509690960e-881', '-1.7419939028246069095236056731585987829852075e+88413', 44, 4);
  t('-7.62222718575298084791963989755171157896421922098e-8', '2.700391860215424e-906440', '-2.82263744683e+906432', 12, 3);
  t('1.58106346989151921e+21462', '7e+349080', '3e-327619', 1, 0);
  t('-2.429317513732e+172', '-5e-2651', '4.858635027464e+2822', 77, 5);
  t('0.1236426444791588877989785775274150', '-1.352448944441565231465913465014929481e+8735', '-9.1421302805786670098968277535229542497707635857461e-8737', 50, 6);
  t('-4.43954145466498120433459608109189970592e+612', '7.4921997112e+361', '-5.92555140785737949234632794113057666075525741786369055e+250', 54, 3);
  t('7.657832561234625660962641308353880838095e-9', '933670.6227621730949083132641808337450043175977092665', '8.201856601827824492429351726267199805326131202106177177883172191743197722895012036050487e-15', 88, 5);
  t('-5222835755032571284423392644738827053269141.769044897870', '518814015064458662316010618618424300078235878355386.8098048', '-1.006687484027137161766303e-8', 25, 0);
  t('7.39473264091755859436653746796717e-5209', '-6.721318739047255408321612e-868447', '-1.1001907405399672166463321e+863238', 26, 4);
  t('1.270035347e-88', '-3.00179606772863413672234360098596983e-398', '-4.23091815148188e+309', 15, 3);
  t('5.42341558926388220256812026598127331278996e+3907', '-5.5796791105079578934372876987929350954316723338048e-137', '-9.7199417e+4043', 8, 2);
  t('2.53e-483213', '-7.966921641312718232192e-2431923', '-3.1756305809267746027072698119990131640262859733199129780927577641050825e+1948709', 72, 5);
  t('0.000042126661266767803321258525491351033371350260142', '9.72490468370304629953429108483e-42', '4.33183281861502274715200268119738320464373364656108e+36', 51, 2);
  t('-4.4016804546058e-4158', '6.230344780997698886503234850021581e-128', '-7.0649066935e-4031', 11, 0);
  t('8.579819712873667561921525458950450385e-173879', '2.3706e-53', '3.6192608254761105044806e-173826', 23, 0);
  t('3.333e-406', '-9.9074541823186649148847716345401e-3311558', '-3.36413e+3311151', 6, 4);
  t('-9.6258606127655262116652631e-69532', '-4.13989779428048150120217031932511436148151629840975e-2524647', '2.3251445062397997427574041686374e+2455115', 32, 6);
  t('-7.628490657079218476257318680041600655852404463308105946e+2329368', '5.693190065891644457394636018303104369077399869e-792', '-1.3399325455129478535630116e+2330160', 26, 2);
  t('7.671288250098857278422e-5154', '7.65464653550507385129773313472822218277275048177557113679144e-28', '1.002174067021e-5126', 13, 3);
  t('-6.550', '65.24407556804507818824129951207996', '-1e-1', 1, 2);
  t('5.16845703096e+864260', '6.38050241889733345267661156e-44', '8.10039193097461926869608786757567756023918792372036351176269746428823309632946149469979405e+864303', 90, 0);
  t('3.4e+13', '-7.7241638463e-56834', '-4.40177094589811847399159435927409788560015621324767443779903703361451052392857887634475048e+56846', 90, 1);
  t('7.3261049958962e+1748927', '-6.075938e-837670', '-1.2e+2586597', 3, 2);
  t('-4.6e-810', '-7647911169041102028645.23085440342151535045257679732464', '6e-832', 2, 3);
  t('-0.658690924922719154496897934506963532092562393', '3.178280915e-896', '-2.0724754750720930358570835596102855247222928153284399e+895', 53, 1);
  t('3.9815061582932841373774251441439598018e-8672718', '-8.76586022531041505298742893882824972430228637623734717706e+253352', '-4.542059827508020484292661296321338357331458332746425418203076039e-8926071', 64, 0);
  t('1.766e+595', '-1.238144981467114374483386903529211573850150543339463709e-376317', '-1.42632730934903495852e+376912', 22, 6);
  t('4.68799944924864471835382215330057183815440997980', '3.3262e+6', '1.4094159849824558710702369530697407e-6', 35, 3);
  t('3.6160987687856049722327795342435359628748991066127660711e+154100', '8.15262479979128916e+8083', '4.4355025008365142935437780656007729100441496442755736312484852932e+146016', 65, 6);
  t('-5.4360264840619213662653828446211298602863088747498635663e+29205', '-6.6302974279241108090102742e-540', '8.19876716415102750498425729926174616102731e+29744', 42, 0);
  t('9.19375245353160279550786460056093972e+53', '-4.88322873029981e+22', '-1.8827200119637124818277812e+31', 26, 5);
  t('-0.00068057022162', '7.9525181690146489435304154080285534016552420821019967139466e-72504', '-8.5579209900041707781076145540989e+72499', 32, 5);
  t('-5.5421150462317965e+33', '4.78372888e-180', '-1.15853e+213', 6, 4);
  t('5.583964880100101519614006016321350750070309274344588e-723933', '7569375.505649745568210769434496142249054374987677', '7.377048312548712794845656020809478472657797031487486220292017445592517433987514021192812958059720214e-723940', 100, 3);
  t('-14436.794120156598569472498872714165315278079740318413185', '6.6320636108595148656301315412050918227799049538458e-885011', '-2.1768178001968215908645437787428641555429323464706365786309184838585674671237694e+885014', 82, 6);
  t('-6.86576762320457954957632017250413494e+805983', '-1.360284335931e-22', '5.0473033040592503577549160627761754571373349303512645242e+806005', 56, 3);
  t('-7.9411065506146760086723365948809513e-9', '3.42103970887144239704323559099e+8998', '-2.32125529850524486434684044554593567945578602178703876399467113e-9007', 63, 0);
  t('-3.723639718833818449506469228915776046300483314070e-1778309', '6254.1094835086163039802618367', '-5.953908751761762120004110355669262627859980279687078574792e-1778313', 58, 5);
  t('9.54488753339e-4675471', '2.28437621186903100394786e+8691', '4.178334323303324720268639443218208362729656466390705136120907341e-4684162', 64, 2);
  t('3.730e-28', '0.000016603702599326119872662740990573447853384148302520', '2.246486877060413191379069423161442689e-23', 37, 2);
  t('6.218344456458442779750475e-749114', '2e-564', '3.1091722282292213898752375e-748550', 56, 0);
  t('2.3965159581727651349084845200811431899626484394e-74696', '-8.7820838878771364433e+87', '-2.728869353526e-74784', 13, 1);
  t('2.4245195050801974604261387e-164', '-3.90714729460362923730e-63588', '-6.2053445193347e+63423', 14, 5);
  t('-2.559217707902385701741133457867299301673784025e-780581', '-2.29673750237058112350514e-2402', '1.114283938e-778179', 10, 4);
  t('0.000096551501836111089013103', '-7e+6', '-1.379307169087301271615757142857143e-11', 34, 3);
  t('0.000865430942545005171975190659906707039606', '1.6789219995971750975218048456704752992760e-67647', '5.15468224701712532995139975237591506850389520048483e+67643', 51, 1);
  t('-3.9e-6581', '-6.225081719160733767991800835415465289124308278e+2724', '6.26497799698892699470412280632425399280623762440415468282701539582804690488e-9306', 75, 5);
  t('885.79618773833589239786405248344813254970337625132353', '-8.36295e+93', '-1.059191060257846683763341945705101827165896455498745693804219802820774965e-91', 73, 1);
  t('9.65741929292706227426625727019365e-7429', '-2.4677357009003e-3116544', '-3.91e+3109115', 3, 6);
  t('2.9517e+521', '-7.89119997561653605330960087176194e-3494', '-3.74049575365042621551648678541640064830712526e+4014', 45, 0);
  t('8.867967317389959920447289565924941031782833389334349e-7', '-2.15597467819795821286944654e-778', '-4.1132056916374029289811136978844562622420666552450392725601270294149736498308822078693437195e+771', 92, 4);
  t('4.2e+29', '-7.2670916568e+6284', '-5.779478501650594456005788464104569046421e-6256', 40, 0);
  t('1.81964562441201581939637324335e+8789123', '8.64395135574162716211962342249e+4709', '2.1051085892603282107384e+8784413', 24, 6);
  t('5.9960913695715898415691301696531290039795025e+14500', '5.89122546044981786106445388225669718e+14319', '1.01780035577076098909745108803e+181', 30, 6);
  t('7.529782e-546162', '9.85281532358565594289257252593829270736e-713', '7.642264421596554309468762746976449164492474436119157625330483881336e-545450', 67, 4);
  t('-1.971426891389372279137085743136216e-80', '1.85913479947398602884855788896892386692178840892e-955', '-1.060400188274221747224002264052111973309156283e+875', 46, 5);
  t('-4.7085518328747609603158779940804238202760068052179512325e-23490', '6.268859274786868560202e+172733', '-7.511018554543712229373369387555499855697154599160758496651837829116552972684558088349015751459e-196224', 94, 0);
  t('-8.564812358541786440990732054002872056425634845601e+13487', '-1.128288391682675565490143555563496097e-22389', '7.5909779996660546e+35876', 17, 5);
  t('-3.0219497137492408e+150601', '2.1739338547946078322037258010e-499', '-1.39008356076903410474939025923850508002e+151100', 39, 4);
  t('-5.077762068268951791099e+477', '1.254827e+657', '-4.04658336828021058e-180', 18, 0);
  t('-4.414134432648238065705595685568e+731', '-0.000243884827122881795080201216906389363908828868891', '1.8099258099496975844e+735', 20, 5);
  t('7.57309768842641e-44706', '-7.661786932306425303078887126145321132645923916e+20684', '-9.88424470079420858030371166638081350563339295871236978392607333711323823153613714037685144159702e-65391', 96, 3);
  t('3.476538981895993084917129e+283', '-4.1578570024018268638582769751912130176633735596389394e+45198', '-8.36137216813309417736884617066e-44916', 30, 0);
  t('-401.8327781843881003083062634374850727', '9.510854329658366e+43', '-4.224991407252707943141843e-42', 25, 2);
  t('-4.374988579918127897401853077450179e-35', '-4.460e+991228', '9.80939143479400873856917730369995291479820627802690582959641255605381166e-991264', 72, 5);
  t('3.050545157e+18842', '5.7857208273730260532973e-666278', '5.272541224e+685119', 10, 6);
  t('-0.077', '8.9691283285351815216604e-10354', '-8.5850037126824644368703490403211252399744760748e+10351', 47, 4);
  t('4.75352044791872245587622456495450866485337e+547558', '6.819501344970945426843487713942e+9985', '6.97e+537572', 3, 6);
  t('2.3794046802606184156e+6383909', '-4e+80', '-5.948e+6383828', 4, 2);
  t('5e-8242899', '-4.6340333152980603418e-636', '-1.078973684434636123202797262876291457371835514544489323948999413008602919202503930976e-8242263', 85, 2);
  t('1.081835133966826259588560869304219194277241e-2014217', '-49858578302402487496090.519165915520287193379', '-2.1698074249234998895175679433883336953413728036380998366881518478532879988065637670159030374377e-2014240', 95, 1);
  t('2.33745385599444758369211913838826364580840764e-94684', '2.173362335414754919677478542441835426383774721030332800e+611330', '1.0755012258682481304453563089276001858000756370506149746287776676100234367527278276983679818e-706014', 92, 5);
  t('0.00213122587016272668', '-7.58628230959649788913899270117964644350805206960481e-767', '-2.80931526561668800323756802149661289612286702824601498737049316481e+763', 66, 1);
  t('5.177219e+3007', '7.57376706609665322147622649022190501e+994221', '6.8357251481e-991215', 11, 1);
  t('-3.438778511509205e-70994', '8.58474724e-923', '-4.0056840526258813882096312016753098953208085366995615819668559047755959324e-70072', 74, 4);
  t('-930181956746305447242.741079684', '-93.89693731745104693920496861473020555987338050', '9.90641424e+18', 9, 5);
  t('-0.00065334084295423337281605069932694588215', '-5.74629048655423902711933339780593769504704527e-143475', '1.1369784463263516087085853341189212311146322428223393365e+143471', 56, 0);
  t('-3.1282343516e-600174', '-0.000015895747997176137898735540296430739374', '1.9679693e-600169', 8, 4);
  t('6.39686e+277', '-7.906309782490789221065222639304572322378123362273262e-5175', '-8.0908289404070694641756770133509507743717844675986060772277329908781e+5451', 68, 6);
  t('4.737506677e+978', '-4.0511567005287017e-982388', '-1.16942074257007270673779175e+983366', 27, 4);
  t('8.03286219355e-5232384', '4.1792100463970e+820005', '1.9221006133624052971313912028817e-6052389', 32, 2);
  t('-313.15844233122798717550199042645348186125070753543765', '-0.000061647130269359434025223', '5.079854341360600191055751776461384370855844602783223715e+6', 55, 3);
  t('9.5829954577883112636092276856688120516999010058291560e-827', '-0.37275', '-2.570890800211485248453179794948038109107954662864964721663313212608987256874580818242790073776e-826', 94, 5);
  t('4.82566933780114581921801210238052742577e-445148', '7.43134277828364e-7599', '6.49367076957213578489282821099579649398500028152598487499724e-437550', 60, 0);
  t('8.26309973038850379189095093801704159994943776522251606e+61', '-2.9327639943e-41', '-2.81751267624954685972460383394207084287015306024705312238154955447312925980298339071e+102', 84, 4);
  t('2.16291e+610', '8.3322564625086376096233691073857300e+7181', '2.59582744450091129871829401711146328244665238776950885484738131274826e-6572', 69, 0);
  t('-7.9706804878661e+2983', '0.000857857628062799904256871927899554790296844936822', '-9.291379160276e+2986', 13, 3);
  t('-4.52081189107e-61610', '4.599e-99', '-9.829988891215481626440530550119591215482e-61512', 40, 5);
  t('3.590093370267e-279730', '-7.94338314583262669315278661903164e-4311', '-4.5196024217344810428662626158420120904314161404134544766543981381097224910556952453892e-275420', 86, 4);
  t('-3.4588464981592976457361749762633608958e+8492172', '-3.521141e-96056', '9.8230843302193739067426580652787289568921e+8588227', 41, 4);
  t('-3.883052074965960868576743506148856564e+74', '-1.850053967655721709377e+1640', '2.0988858394689606685040972119186668540423783e-1566', 44, 0);
  t('6.0053221370669650064718306816e+919', '-6.2550557938528524724323975796103534127e+553', '-9.6007492e+365', 8, 3);
  t('-9.981988992266175326096983489e+785844', '4.9262608738395023e-726', '-2.0262810370588970908199108167273414274913326131390490495456873672336640585632246896659299548241e+786570', 96, 1);
  t('1.3567442371744515525949659296916525228583601957509939673e+6350', '9.20e-633611', '1.474721996928751687603223836621361437889521951903254312282608695652173913043478261e+639960', 82, 6);
  t('-925.132495132060664105582168942832170018704128335178427', '2.816905877176005772911736153920553399863448721846484035787e-17', '-3.2e+19', 2, 2);
  t('4.25802451194307177337299273323929680282896657712757298589e-91', '-5.58385255192431028361423347018637947437431001891313252e+166754', '-7.625603420483710776519206081097236371938875933318402963771462011031121583425021086602836253169576672e-166846', 100, 5);
  t('-4.624166e-70', '-9.3289351920', '4.9567993611590736410381100222783067652e-71', 39, 2);
  t('3.095934977386501436397164003e+8543046', '-0.8113213089353323344967667936292695786579663', '-3.8159172491712132469758486541178761750384395407780130776116031191500742554815487e+8543046', 81, 2);
  t('3.57895373869154219312736490614832144495301e+5024', '4.73762127542716365462398965067586e+5381', '7.5543263815845e-358', 14, 2);
  t('-9.953070670236959435989542552e+581594', '0.00000233495966134446698866063053490907985902957073882564195531482', '-4.2626306719602998249171628986000174657055786192572518354967928005561185698616724356182389674e+581600', 92, 6);
  t('-2.280332344157424082436289042722055625046720019119092679414e+2600', '6.70489912007486008818693915571900115e-976566', '-3.40099426303070792419457e+979165', 24, 0);
  t('-6.398e+53469', '7.98315991521555996711172647379199e+8622698', '-8.014370334490840848235213e-8569230', 25, 1);
  t('8.5952018097e+828135', '9.74749e-4369017', '8.8178616338154745479e+5197151', 20, 0);
  t('-6.45481980644574225e-726', '9.84161678749869714232432560603044784e+427640', '-6.558698581563315879188760226492731345323598196072329499159200289768400292595353610439103749518635909e-428367', 100, 5);
  t('-4.9674477463727121938073542124e-16695', '-582987902987.1962384145646624091245', '8.52067036197457569830597616633680114892907226669e-16707', 49, 6);
  t('-9.14717211469063496048017806673626480487794e+89', '-1.03e-5696782', '8.880749625913237825708910744404140587260135922330097087378640776699029126213592233009708737864e+5696871', 94, 3);
  t('-4.8635400322120376778537175147648e+743', '-2.36707469602975260e-75339', '2.054662677257104235075798298889121242810358e+76082', 43, 3);
  t('-5.63548988694627886787587349987896646686700785196658157584e+40477', '5.77209e-747', '-9.7633437575406462267148875015e+41223', 29, 5);
  t('6.83695e-558731', '-3.94342139703431498783160018151252988e+637969', '-1.733760943008e-1196700', 14, 5);
  t('-7.98262e-86673', '5.741118846201109301e+436', '-1.3904293246e-87109', 11, 6);
  t('-4.26874941731106819593450850675989330588172453740311826632e-20063', '-3.2403577474686503538650654271626333181e+3762', '1.317369793704350008e-23825', 20, 6);
  t('9.2882326610961883e+27', '-4.30662480016486641955935993893450580735e-32370', '-2.1567313365075628892952376606705082040079243923653220981290489607445066878641885316581063574733e+32397', 96, 0);
  t('9.666240186989400476881729679425669361845280068784e-81', '-4.92037839801832192380005136e+6049161', '-1.9645319e-6049242', 8, 3);
  t('9.232486226130084335699618763355315307990627088319426399605e+8844068', '3.9e+63', '2.367304160546175470692209939321876e+8844005', 34, 0);
  t('8.1743684175973623740577760207067029620194414583e+42788', '-3.020710372345526921392741874806840122036278485855143377e+690828', '-2.706107971301503253744450452957962096033375556282593804502133202e-648040', 64, 4);
  t('-5.8537208685979961804257724819363504650619833457e-406', '2.70006233530852421326150389e-2810', '-2.167994713325430435716938510175728970929943257751040650986980622372211e+2404', 70, 5);
  t('3334.78107253269059738584567915642232492818', '-8972180632559.4892', '-3.71680108671795584813762107152084186283616736293492881508957866299771835729380542351661992694e-10', 93, 4);
  t('-6.242100006184550179391666553480790558550805652e+556', '-9.82836522790083710919500145623437928471e-19562', '6.351107087946253510204772800761927631308542246178465075452e+20117', 58, 4);
  t('4.66473793450477093405098e-46', '-3.852341070586643781632611090691822273760467745e+70646', '-1.21088394019e-70692', 12, 4);
  t('-8.162327659811364194903775226733127923186e-5345495', '-3.9499236279e+297', '2.06645202002321078723046548621931342106646203289747408840020929357574427749355e-5345792', 78, 5);
  t('3.605662873886263663818e-8434', '3.30377906613266298e-124080', '1.091375301347550388541276612640161450663677797598174358554637487080243306482718e+115646', 79, 2);
  t('2.03535220477482354809298305754935022943376057531e-85928', '-77546278827463270.20383055638116849129227', '-2.624693583690049169943640617270614776575433786581076462305222301986434947657494818191072613273e-85945', 94, 6);
  t('7.40e-591592', '-7.75e+8851393', '-9.548387096774193548387096774193548e-9442986', 34, 2);
  t('6.8056650204973653378574186579424614131830798174e-638822', '-46.622459', '-1.4597396118676119888608661027386954028278688212048189049831112511675971445435771631e-638823', 83, 2);
  t('9.316220168516861881103685684687558948260e+575928', '5.70085284063452695599704573572461275330557667344e+6237', '1.634180083041738465127651313886726776260150759121302582819452585393977e+569691', 70, 6);
  t('4.6967739383923396529003234520007972819053916837e-552919', '-4.447393368690689980558657011961109601730e+50', '-1.05607342302061019777230189726663e-552969', 33, 4);
  t('5.450138739805869994519508442987338921445437908601849112045e+518492', '4.06527155698e-80', '1.3406579765743022303223209960814397180422027838872e+518572', 50, 0);
  t('-1.2681888006403972113682033787e+43558', '1.9862908180410938e-5896128', '-6.38470856896525218528734572580007711248107717486e+5939685', 48, 0);
  t('1.37781586723665857435505727168797934320870e+99', '-8.818473958990057520654237407084886e+5158617', '-1.56241984003596693849913085767307745854104466211882791063779148459391961491e-5158519', 75, 0);
  t('-2.237687692624590743731212540513663805303704545e+74', '4.8223370446511479025767805692595965616237395703771e+72474', '-4.6402556932568512262677842519047e-72401', 32, 3);
  t('1.424005635978500062890684347838676663498465572e-81342', '-1.19460163218743874946880467e+9870178', '-1.1920339e-9951520', 8, 3);
  t('-3.3537956017922014741624740e-765', '7.7873218541505224509926023073605323645e+44220', '-4.306738137456948875377946472291268131032224e-44986', 43, 2);
  t('-9e+59210', '-8.835655602951134414622109234388728872915343354967232709e+125', '1.01860013613409446193397447087244985344864095534013e+59085', 51, 5);
  t('8128823887199812657410247368065460.20360163414228', '0.08378484866', '9.702021328685196e+34', 17, 1);
  t('-6.00859143064029334973412510864364391574678988749749915712449e-43', '9.59071377531e+3708057', '-6.26500964517218954408196509104673e-3708101', 33, 5);
  t('-7.9621423782632874685119197214509e+4326', '-6.07930777589e+2675', '1.309712005343839296531734280805951873e+1651', 37, 6);
  t('-8.3512004035417319093448522511292813852817411e+32516', '-1.790e+45712', '4.665e-13196', 4, 4);
  t('-6.83e-158', '2.8175895559853748461816585e-27970', '-2.424057821158197187219767074023250518311886226126555993e+27812', 55, 1);
  t('-4.2410547403160631485383629457023185296339353385e-580201', '2.1918565642234396487048e+1925729', '-1.934914e-2505930', 7, 1);
  t('-7.02824208339755034174567248094791164180597828587379234335959e+974220', '-4.659990013559070650618250297109844154e-9', '1.5082096877777910744915140833790404573363236163698256319748698307926e+974229', 68, 2);
  t('-6.464231475077962521236042754690e-560291', '-5.383e-816', '1.200860389202668125810150985452349990711499164e-559475', 47, 6);
  t('-4.7429026700504370681e+73', '-4.4919439983271391345e-49571', '1.055868610965933312279661988616829847461366643593609978489232372194518375246873e+49644', 79, 1);
  t('-9.126340058738e-2028530', '-2.2702873716958548283767243705751672333942550136640e-61307', '4.019905221038526202647646276192742677227481870128713776738013527397e-1967223', 67, 4);
  t('-5.60966468517303883042747184299310307195911155058e-70028', '-3.801205231009667327576458080e-460', '1.475759487914577212164668726373048195407577084818334116296294690839368062e-69568', 73, 0);
  t('0.0688564910139141584898941610196660443328', '5.429194425261882771686899930228932516786965978e-8425928', '1.26826349584253090908028349363034170653783057416832776839847699116577108526703435844801317155878e+8425926', 96, 6);
  t('-5.007847896e+61561', '9.1904561e-85', '-5.44896558071802334162719084203013602339061279015303712728686e+61645', 61, 5);
  t('-120646117.83189454698870016934788021124925395', '5.274802609769103166e-363', '-2.28721578336323102371472756402809367659579855197299786e+370', 54, 6);
  t('3.561126432513e+516', '1.491357423223653286290089185524875706766793345e+584049', '2.3878423623060286390699561437183025363357073935830038523234525524927636838016362437763979188998006e-583533', 98, 4);
  t('0.0000428655126657096175785391128804551661098359211821992803', '0.00000793983755664915807204546612311444442002550375501', '5.3987896301243861003330218819639490714046381445221466096e+0', 56, 3);
  t('2.661588631416e-15850', '8.094740032625594480114127e+56588', '3.2880470783355005090592053672745e-72439', 33, 1);
  t('1.598566217012987004339943101673162639730215928921906639e-635', '-5.5152130078801634771423297e-35708', '-2.89846686742e+35072', 12, 2);
  t('2.703209117229891481438995327412080901452327075200757612e-34985', '-1.883761695300124878143165356512865376593709417888891529e+5144', '-1.43e-40129', 3, 2);
  t('-5.534342773127e+71', '-2.3090e+8498', '2.3968569827314854915547856214811606756171502815071e-8427', 50, 4);
  t('3.27e+439', '1.11341826878604717185126474711666056503463301846874978e+54', '2.93690169424403294345291e+385', 24, 3);
  t('5e-240', '3.38821907290638529740171e+74688', '1.4757015093805734438e-74928', 20, 5);
  t('2.03316186915365377e-630389', '-8146428678.19829265722968550184876355949713211', '-2.495770784312959362699322831209217182948509522379341695956869850581819979308406e-630399', 79, 6);
  t('-8.80968952319354396523912357e+2456695', '-6.492980183108946861591227042230148725400050615256e+614', '1.357e+2456081', 4, 2);
  t('6.73e+4224706', '-9.233799954672042747306446103052e+9209532', '-7.288440331215e-4984827', 13, 5);
  t('-0.000003414722628277670669974879878851201242852897449127509002623', '-7.014149958310288e-8', '4.868334222355689335058624170936971298180118484017358304397577854928e+1', 68, 6);
  t('0.072', '5.1983683086560884587699109613617418155e-9', '1.385049995016875722967743429414378372439e+7', 40, 1);
  t('-1.338978994e+8067997', '-3.97070916485840e+5074731', '3.372140689250781509100506146409902491645127e+2993265', 43, 3);
  t('9.56220432914296527994334833884e-291911', '-1.3300786550684799774706702e+8422272', '-7.1892e-8714183', 5, 2);
  t('37129207042.34059433', '-3.4143681924e+36210', '-1.087440046008688747354791577515399771213e-36200', 40, 1);
  t('-3.8252411190396990353e-5554705', '-5.558332697645348325983931729040156241965445418932890464e+1891319', '6.8819938048331811059310395225218740856654538852094678988262914449015997267270979957638614105e-7446025', 92, 6);
  t('-9.493901598022e-7007008', '7.831756437e+943619', '-1.212231467409971498325848664284757301882369557657887107757613223640141657791e-7950627', 76, 6);
  t('8e+3736', '415529.64400479270780702', '1.925253737109482425719934804193610266756072568e+3731', 47, 4);
  t('4.8261982665872253349e+8954', '-5.633701332273250863710377840063314953449559081e-6349328', '-8.5666562388386490227642480002993e+6358281', 32, 0);
  t('9.758536600372596325785494255136077167427367e+14592', '-3.585215106457210755904915e-7', '-2.7218831536207752639190418827482157555625712612874502608548806056775718717619968e+14599', 81, 1);
  t('5248.18337176732283137202642608365', '9.0757472991007097541861102207041693219054e+276847', '5.782645989148850083866552505497745692741757445028131910249582e-276845', 61, 3);
  t('-8.0883700308754e+523', '4.082408241691223176384316789368180e-4525', '-1.981274177401871791343969565861270460864529e+5048', 43, 0);
  t('-9.4263305836241369821327693986430750344739839184327753e-395', '7.80053e-4202', '-1.208421810264704703671772225559426735680009424799696e+3807', 52, 4);
  t('827369713437207.8', '5270746.693443841509271', '1.56973909307072872298433504431718708411091593563381731e+8', 55, 3);
  t('-7.34972710593718317572113e-213', '-9.27768949044257177207565537e+569431', '7.92193693646302513719601435836648594e-569645', 36, 2);
  t('1.12753491736721481618528e-6356224', '-9.95603226182194995745e-3889', '-1.1325143267071699637599563469e-6352336', 30, 0);
  t('2.53415295977175651933249676513e-969125', '-5.9177478931779603437e-13', '-4.28229286802358325518944712427e-969113', 30, 2);
  t('2045768316.447', '8.667e+14375', '2.3604111185496711664935964e-14367', 28, 1);
  t('1.0745413851003e+670334', '-0.0005812493288670238534450263', '-1.8486754852600092072717422551e+670337', 29, 4);
  t('1.0342009147082649505352372932771459593766933290995e+11321', '6.06616927193404744763816689964266409963740849048647890693882e-43039', '1.7048665613291329544205259224167695273692375637911853156678814141196410111663047887e+54359', 84, 0);
  t('9.60056356103035832418236626389542455410e-187', '-6.26348208e+436', '-1.53278375165884697864137040269067466607967049536126396964e-623', 57, 6);
  t('-6197516404.5727345', '4077346541.2387566306616005827481760152', '-1.5199876541987131942033977753149629852393423205812488857917610969656973883190681815e+0', 83, 1);
  t('6.163639583583888177274965738338099883087298530097564809998', '4332065.091903051587081610175387143489927575548196714000264', '1.42279477635370347587358751666663337470283118e-6', 45, 4);
  t('6.6115091139797029905176079494224764948219022e+691017', '7.6055417816055105231823229703355815267288674655065e+4880439', '8.693015308876562683032125349295039508925e-4189423', 40, 5);
  t('4.081743186005789617453227486530252062883326e+3732', '-9.636212118238603608715858526409473e+53323', '-4.2358378332915823824929359201722e-49592', 32, 4);
  t('9.996147e-107080', '3.414026574568310e+6122', '2.9279640277152713115961628015455467038358355650992235920418058938069417888e-113202', 74, 0);
  t('6.077174e+13', '-9e-8468', '-6.75241555555555555555555555555555555555555555555555555555555555555555555555555e+8480', 78, 1);
  t('-2.691e-91', '1.5631861078143808079652313930129878e+9055', '-1.72148408084467219701462721427665493327930551598941271394e-9146', 57, 1);
  t('-7.1186912e-554', '-3.3021539378796648848498e+10266', '2.155772060878227621617125677860322420008775232683732837e-10820', 55, 0);
  t('9.298831373664187864963e-875910', '6.092526889598348072692618e-39198', '1.526268417385223311898121582721796944492167527e-836712', 46, 3);
  t('-3.959199513713537470721417013508925579431591685403665e-23', '2.177878829998357051791642667058828071220900322e-344', '-1.817915422648432746650098260870105271574e+321', 40, 2);
  t('-6.260661890e+975', '9.6354947098881745806e+7470091', '-6.497499172e-7469117', 10, 5);
  t('244.4549960429', '-5.72687888477033537235946280197051727843316295863216378e-6816', '-4.268555367793557970295249887962477722e+6817', 37, 5);
  t('-6.1423370070253e+74', '-9.5379e+13283', '6.43992598687897755271076442403464074901183698717747093e-13210', 54, 3);
  t('9.19510650518108750615390751758334209738009e+2057', '6.69005112058700081722e-9', '1.374444879335135370200330960253771661552962647029987667998755566016991577308754896885437262139721e+2066', 97, 6);
  t('7.3215842477188520866324170442227693187792985524973e+9421', '3.24858883704858393023258050e+326349', '2.253773750688214563968520941574608349783978250431353830408605841e-316928', 64, 2);
  t('-8.71579276054510e-840725', '-4.5318428404942247115517113500e+4076272', '1.9232336749777029760673710759179490621192e-4916997', 41, 0);
  t('-1.434965893134070200e+737', '6.56713666493874217451706755585572060113737727797e-1753374', '-2.18507085560622403906384487e+1754110', 27, 4);
  t('3.0712e-14392', '-7.759710318697827649447486735203e-695', '-3.95787970666846253106476319638807542737616396452358012718290267156312677821860868274851929e-13698', 90, 5);
  t('4.515454616130906017e+7619844', '9.884728661928585523753677730e-160', '4.5681118527030023482697598985949953012524191993079407720918526810486454082388753e+7620003', 80, 6);
  t('-2.8110373e-8559', '-4.437402446799e-59', '6.33487120832998204565922852639953595216609541e-8501', 46, 1);
  t('9.898174463814491001960787200757e-351779', '5.17618853629777668395280867942999428217292400609474171e+9746', '1.912251533035169003660476364567518725666095003821083516178081878907004775e-361525', 73, 6);
  t('-2.36612760320894668924700844852e-11', '-3.2745317602179171193943683425524684e-5640419', '7.22586e+5640407', 6, 2);
  t('-6.9596693490802999514161313229487402867972607764597969656', '0.285226683055', '-2.440048481627601891131e+1', 22, 1);
  t('8213460144.175071054072894126373913199631949943627338', '-6.406146817044620719196002718063148338027967572686073281975e+3322', '-1.282121746308060279295936331964983994097358126750768e-3313', 53, 4);
  t('-5.003483249812549e-4611', '-7.5567e+11', '6.6212543171126933715e-4623', 20, 0);
  t('2.449575143613357647196960436154e+202032', '-5.57899958218e-96022', '-4.390706805997256525137717457355e+298053', 31, 6);
  t('-4.46938143438468080209676025575587201656881822e-9659', '-5.984e+4', '7.468886086872795458049398823121444e-9664', 34, 4);
  t('5.6387496904523695598823606916050957932186171021418578327e+9365', '-9.544373379e+557', '-5.90793074258706613397501775543759958054195178625341353873704106269332068635953979059e+8807', 85, 5);
  t('1.974211512831110190393842525e+2159', '-7.5333111e-954', '-2.6206424859197839186461349312920317335626826827847319354699157452823e+3112', 68, 4);
  t('-2.75661977053791701e+2119', '-7.053741385645802389176689645132526282432285239880e+94', '3.908025003790999946844454749816e+2024', 31, 3);
  t('5.1342e+9743616', '5.275918495265529345902796796716985476235726904590e-53416', '9.73138611714206009726988993150647502487193836013279e+9797031', 51, 2);
  t('8.699730749512133925496377234206818932534723e-26', '3.4453652768954455250050574257269430987581619e+18096', '2.52505323828865520349873985458855189292005057179257850602e-18122', 57, 0);
  t('5.664812257591541352640567193767046909e+5285462', '-4.50890863902081e-98996', '-1.25635995562e+5384458', 12, 0);
  t('2.48787856672556002496774233785963666018036490e-3933029', '7e+9', '3.55411223817937146423963191122805237168623557142857142857142857142857142857143e-3933039', 78, 2);
  t('5.9539641341102645550947770478494e+92', '-4.75178622046245861e-353826', '-1.25299494924054185512831751632966705933995697937245501289820563149146969666625336886948648e+353918', 91, 5);
  t('-6.6467837002751853908074751004e+306', '-4.038815721317747800760251e+9875034', '1.645725915443979128881e-9874728', 22, 6);
  t('-3.90481922628843248465563624942610858391358568576661024620e-9430363', '6.86298535478499436869820023137287780e+42', '-5.689680254914027591083142859442789256657e-9430406', 40, 4);
  t('-8.2518803333e-533225', '-9.24020999585292426696466624169357039545383424430979282339e-18', '8.93040346161343297196403649076651682277186476894263236646878545305731717653e-533208', 75, 1);
  t('-6.41067821e+1929563', '-634707603.4845767020225598693480791090464897823639', '1.01002070477887043351480601517769897426205015629537362848124473191484020741186184361711283e+1929555', 90, 0);
  t('-5.29769762830211349249371432620016208443703741565651e-50', '-6334138.745141023440', '8.363722111969881402006309448199286039523090527712710816906890818173036e-57', 70, 4);
  t('-2.3246540870049496042975e+2618120', '1.84626e-2187859', '-1.259115231335212594270308623920791221171449308331437609004148928103300726874871361563376772502248e+4805979', 97, 0);
  t('589439697404682003466316343821213808907714760264392950296.63', '5.0538982896632129382945e-19414', '1.166307004259006792096343990731e+19470', 31, 0);
  t('8.26480049309339941034744554426537640559003873e+24071', '1.434717967251003e+3742407', '5.7605750271108703532514803677705634244967e-3718336', 41, 5);
  t('-5.30821970682964756e-76', '-526439554137776.72963094920296618803361', '1.00832463387437845435615560827809802074792353644794953149743469241120147e-90', 72, 0);
  t('4.311965340061586939626215684855913788626e-8', '-9.32810863172e+10503', '-4.6225505e-10512', 8, 1);
  t('2.13266501674148824', '7.37536229399840276277838997115967667e-146', '2.89160712617e+145', 12, 3);
  t('-4.66156710e-446412', '-6.820532155030260614e-8', '6.8346090804102631685015214277836006333821193730158535e-446405', 53, 6);
  t('-8.82064291547559016172450119651e+79', '-5.4674580800399448262374442512877928110e-1791', '1.61329e+1870', 6, 1);
  t('6.3596711367398199051223744060326168881439e+3056', '-8.883549974554361168e+2140706', '-7.158929881586e-2137651', 13, 0);
  t('8.63344450591553536794379391357748211859613547151595216686882e+631007', '-9.306113242e-9', '-9.2771754237326476839618431043131315880414852483652510521092367339150846752612512428e+631015', 83, 3);
  t('6.32284303943007431017430713697567343622998485776636892e-18204', '-4.31887792782567090607946548286754406305e-12', '-1.4640013320805515334524558592826803332326816100722554340228220377e-18192', 65, 1);
  t('1.5815379736946253094296859750535722520523348159337501e-17665', '-4.11605663823501643028e+647', '-3.842362029237760670080940768743697502e-18313', 37, 0);
  t('-7939.872355277727171649400758601209134330843696485138807139', '-2.91112674291440540550e+1323', '3e-1320', 1, 6);
  t('-0.00000148115859547900950011238279210334603860165011296461564704918', '-4.4295118395258202244277902683985985624021130912914319919e-7', '3.34384159956905704091232369069871203206530571769674042045243027875e+0', 66, 4);
  t('266981379708464635528679062258.658592890445', '4.964727712220529955255959094237e+36', '5.37756338683585589150289962166402092299189400317172648438999576e-8', 63, 1);
  t('-3.187e+919', '10981.102917099818918109034451475537138036009209', '-2.9022586e+915', 8, 6);
  t('80.135553297053310740901117923616751991741861777471022922438', '-82833381935785564468926997.89', '-9.674306592863288431519859031141859534552449136845119431629912988063819273790784971905634170315096e-25', 97, 4);
  t('-9.447e-2214870', '-4.02859195e-248', '2.34498805469737385539878269378957578466094090268933789633373e-2214622', 60, 3);
  t('3.537998468838195685676415551113685395e+48', '9.69335109087128902877435769874994e+97', '3.649922958191522497244e-50', 22, 3);
  t('-4855720230870.4513269267507827161759675488297126143951243', '-8.978021104558769533582361710462833003030e+3869986', '5.4084526805187196503132422238527776973593807115231502537719311625302164553227775876408569e-3869975', 89, 2);
  t('7.9073150787460439055865282028109202712084498e-544790', '6.713853057839122093518452955731930140287889480527347558e-575', '1.177761117293657583125510758944553850835483047228170334416596532458195983379713435e-544215', 82, 1);
  t('-7.0246617720278760567399028884942086203916538845346940146531e-58891', '2183595774.1495622228711848208567', '-3.2170156469385e-58900', 14, 6);
  t('-4.1554766870736563308759442357139307684550517467617353e+4024363', '4.9462375307766183668878880051431621794434865e+640537', '-8.4012881735204432464215822e+3383825', 26, 4);
  t('-4.29667656930e-8', '-5e-767928', '8.5933531386e+767919', 65, 5);
  t('-6.5683068343471e-734', '-2.4572504884364226184e+103947', '2.673031042320227954491762381864192357452574244095247807952231531984e-104681', 67, 5);
  t('6.21382e-75932', '9.82665802304965078603145627378041998277e-4604600', '6.323431613702960816871350284176133492595459728316355381847081047538451474732340194023040634744e+4528667', 94, 6);
  t('-0.000009258864551042', '4.990762091803464240668236178624803445974525855403435282722e-9604371', '-1.85520054467196855669308613647928e+9604365', 35, 3);
  t('3.283065692310497267366594014029117947915679163022009e-4099', '5.0673e+72044', '6.478925053402200910478152100781714025054129739747023069484735460698991573421743334714e-76144', 85, 3);
  t('-246205519.6160710857', '2.0786429621526408051589737776', '-1.1844531461098103557356950933e+8', 29, 5);
  t('1.96996176661e-32', '-0.007520124374695263931707081680897322788587342963575305', '-2.61958668295273807174439251980211589593757178491479789069225901928231849254098778e-30', 81, 4);
  t('85309.282645332177647', '9.0530400989665391896526077147820833230375706618634454476763e-468', '9.42327458099636194382951e+471', 24, 4);
  t('7.46549264710535489902e-22021', '8.219797516276011666e-6526472', '9.082331568780059900776833452013563141000516755655840228779e+6504450', 58, 1);
  t('-2e+9408647', '6.792365459e-279427', '-2.9444824e+9688073', 8, 3);
  t('-1.4869439219432551169022982211734456823791811e-53821', '-4.753962373314970926600160e-174260', '3.127799097211614676184374590345322354170231441e+120438', 46, 0);
  t('-9842810715.9', '6.60e-1698463', '-1.4913349569545454e+1698472', 17, 1);
  t('7.4395147817040973452e+40720', '3.84413418436305e-7547520', '1.9352900874184e+7588240', 14, 5);
  t('0.000332961362352164511623710207423', '3.56408060371164469639468861134271077818745896596463780230e-383', '9.34213895177952784485746495343061820660593602008243425404005852292077e+378', 69, 6);
  t('9.11885341940473839408575938384012774321e-67661', '-7.9088914133010387e-82', '-1.152987560819055655134282968842532883671228044459732713502144508654141256434344544533783737e-67579', 91, 5);
  t('-8.5299488303848936302456362584099035594107537410287271e-8718651', '-9.00665e+994', '9.470723110573735662255817932760686e-8719646', 34, 6);
  t('0.833089432216403153611265477411666667722437374885897542', '8.452416810975134215117323697872422419111482142244267e-3703', '9.8562275245899960143964693715285659594591676668928363960727587189814679235958067247966474e+3701', 89, 4);
  t('5.267800714111572978525196102401627', '-7.1623611695175815037718380616992824807e-3141', '-7.35483814545809056755308e+3140', 24, 0);
  t('77089446871073734.20656227941369002814763630781', '8.8360092452637473565059972961820798580628476486164e+369045', '8.724464261102375048586255640858e-369030', 31, 0);
  t('96.057686032', '-9.7664635750735133301862997579178533878422944783e-153548', '-9.8354624776529678911279137666105749233297209023387053186670156e+153548', 62, 6);
  t('4.6073647059765138615100569747306494e+460449', '-4.016357979536216937024597732603461e-74', '-1.14714991279451204739420697127464043074572817763e+460523', 48, 3);
  t('4.350908566861098931216756910746119162724e+35343', '4.2609754978301988244448e+850', '1.0211062159537637329667714295253130817127687109713938473797596231e+34493', 65, 4);
  t('-9.696817050283784414e+276', '-5.58646555503374552833341e+848', '1.7357696e-572', 8, 4);
  t('-7.331827097971634879917964286056959941446213902723568401e-304729', '0.00000831207514828107390', '-8.82069395088163e-304724', 15, 0);
  t('3.93298669402072794114299899887e-1512', '7.72901242e+6084', '5.088601855320511881313549e-7597', 25, 3);
  t('-2.96069842842096310574354374361594164148796243762076951755e-52360', '9.4498830587855422289076008000990209e+3302491', '-3.13305298065927512091787709367118789213584665501097002376246990602481396197340165632300247435479079e-3354852', 99, 5);
  t('282915.480593920130121418412422920892712008902242859399312', '-3.354915161e-23', '-8.432865423326874110525932683128164882736624952849572e+27', 52, 2);
  t('6.140988363085712725567739450374627167244289442639108773e+6386792', '9.1120851642928672190039064249101487287727949738e-1704960', '6.739388682570853684179467563520673598429275815020203947894675160880482100515988907105380019e+8091751', 92, 0);
  t('7.513800104665443626e-9', '4.351697135827428703651877809027189690e+429', '1.72663672818691570922446114251232631073214343999185786701640482832705308105395741077e-438', 84, 6);
  t('4.99868913267e-21296', '-1314559598.28304496971968258586544140117578601', '-3.8025580119751292422e-21305', 20, 4);
  t('3.122011910671616030999170882e-19115', '3.11255681556277824063788823568894500609528e+3240', '1.003037726110431890768655418563117605342573532007920614522264041479690874526694135e-22355', 82, 6);
  t('-9.19840826446854973629e-95', '-4.0521142148298574429375388605104558951973577402808e-43504', '2.27002689874939221510659964466029590690036233697910568845721348412425768805251323e+43409', 81, 2);
  t('1.1673464029850883003119121721172557888837420940492827e+957504', '4.71e-592', '2.478442469182777707668603337828568553893295316e+958095', 46, 5);
  t('0.0003131148567813440028672576', '-6.13903486298707232884630445312e-2634', '-5.1003922240146979403385557685681305352589500148346590588099546742473128356912091213547398e+2629', 89, 2);
  t('7.45962530480932527632954615098339718750060776e+71', '6.04e+779', '1.2350373021207492179353553230104962230961271125827814569536423841059602649006622516556291390728e-708', 95, 4);
  t('-8.007e-2746151', '8.2348473447307518277589610315123817900739797596584615055354e-84', '-9.723313213721508799981483924981536944935804447e-2746068', 47, 5);
  t('5.52206533635738356000288e+42829688', '2.97651439558070354509685445493557826e+333821', '1.85521203746103682221328466556500511430721560358069905849521383076460210798e+42495867', 75, 6);
  t('-5.088e+875712901', '1.4199149740929254e+2656459117', '-3.584e-1780746216', 4, 3);
  t('9.2e-326014121238', '-9.07e-76', '-1.0143329658214e-326014121162', 14, 4);
  t('-7.22327739417772967251812419997120919619633699046547e-6186894660293', '-6.3769027162794938112644712016423387e+15531292765228', '1.1327250415374127268627640555328593529976903962e-21718187425521', 47, 1);
  t('-9.12342611074207276438290905850859777332891323899289165355751e+23642610836', '6.638794680948338252998590908e-512696438889772', '-1.37425941743973470637312280491167133324e+512720081500608', 40, 0);
  t('-2.718876507122563373164805019732402755726e+610154533', '4.308847138979439886664436527015146639e-5717273', '-6.309985987960889637562597121517139258568481214838008e+615871805', 52, 6);
  t('-2.95e-84395067694', '-9.051e-25', '3.25930836371671638492984200640813169815490001104850292785327588111810849629875151916915257983e-84395067670', 93, 6);
  t('6.12234136045360125959997290809070858536820218876929808e+880', '76.92743310785533857019259359198418220218', '7.9585930702638079165374686911356231401285147e+878', 44, 6);
  t('6.4140648917e-84014232852473', '-5.510e-848230403796', '-1.16407711283121597096188747731397459165154264972776769509981851179e-83166002448677', 66, 2);
  t('2.623435946859656088898662e-439110203108669', '-2.59934167915826734419163896376463735432272483690964181443e+22771179081', '-1.009269e-439132974287750', 7, 5);
  t('3.77771400014986649e-3640965368', '-7.768388036705130880554826e+60288744811328', '-4.862931643347902095997739979134e-60292385776697', 32, 4);
  t('-3.7190486e-9449640603576', '3.010673196779523061481248096425808318766627214198e+460955994', '-1.23528804254749956038144330221809830464007779452844731e-9450101559570', 55, 5);
  t('2.62926579885660625e+4862548', '-8.59856524321313877484710204275252176986556420841500574858e+4819840', '-3.0578e+42707', 5, 3);
  t('-2.524950163943407096544e-866668', '394.90557961', '-6.39380726511130060491271669525652058689584236538105772650417477852e-866671', 66, 2);
  t('550.4642428428653', '9.1101038960828308023431523605217648626593940842806e+17458828824', '6.0423486836364e-17458828823', 14, 4);
  t('-6.8e-38', '8.9551e-1822287', '-7.59343837589753324920995e+1822248', 24, 5);
  t('-5.447540e-28798686244', '-5.345151786051220e-306831268640896', '1.01915534264452015382856476e+306802469954652', 27, 4);
  t('-5.024994880550e+2137246', '7.895448984359022572465424e-386506976500376', '-6.36441941491177264370878071531473745226066140771862199179510949585079669357392134354992036344e+386506978637621', 93, 3);
  t('-6.08564742269871323619829704490073949296602829e+19025916382', '-3.6e-43810970160450', '1.69045761741630923227730473469464985915723008055555555555556e+43829996076832', 60, 6);
  t('-7.3315478778634806938360668122298015018709306244776522955182e-371912096', '8.91746537243695600952885083953e+2702', '-8.22156e-371914799', 7, 4);
  t('9.73260146343212982991740729169222462263658685696162754956027e-22368507555', '-5.41457945052064335762744916138190772711113115303334672e+2471502', '-1.797480589650648399009368412431136314816546487012269e-22370979057', 52, 3);
  t('-0.00009077766680012965169574', '-9.864458426424591130875595041492160946022032e+901238013227', '9.202498796787199549471832328e-901238013233', 28, 3);
  t('9.781679164322202110e+9597', '-8.641207387556310909276425467620158738317e+729052237', '-1.1319806047483846773794769631121259324922385e-729042640', 44, 0);
  t('-9.8262134872771847603721e-884906848471488', '-8.40191682826069e+27069057754468', '1.16952044255256e-911975906225956', 15, 0);
  t('9.68468476476931178176001218812114617858e-9042', '-4.351426837802378087e-3922', '-2.22563428635293669783342668520694795499991487062628653382642814248e-5120', 66, 1);
  t('-6.673881203135270007501032890781676e+197', '-8.6130247e+5574477675', '7.74859173e-5574477479', 9, 1);
  t('117689.2203859123922039850242', '4.672903266480619296129041823997391978768000422443130e-159602702578', '2.5185460445995839266693837903e+159602702582', 30, 0);
  t('396858.23797993900145', '-1.1351391313670681683112832752941408479772854e-231501514616', '-3.496119788435058122899220192e+231501514621', 28, 4);
  t('-2.3747155e-943', '-6.252178142e-495179', '3.798221109612138751500084816e+494235', 28, 6);
  t('0.00840506301611388376116385269076480', '-1.9099e+8284284716', '-4.4007869606334e-8284284719', 14, 2);
  t('-4.16692746895958397175358959429449784278165607548e+340931500', '-5.717754287e-87', '7.2876994354822018810434929665498895244528301212045420656951011088455e+340931586', 68, 3);
  t('-9.28389313299006329424285545887e-3925', '-1.97555681198482273934125909e+5297566071', '4.69938048689302234899586964124151630183496133717869393459635207158684409706031e-5297569996', 78, 4);
  t('2.59540333821819953654940688e-246897272142', '-4.9195761293882986672634895403121365796657943e+725', '-5.2756645490531572282469478803463890191217367723692684945787639199827846880769858499841263419547e-246897272868', 95, 1);
  t('6.9395240e+2278489037670', '8.95379929543273201870087684015352822705e+2686557018', '7.7503680516267555657047785558173852295405088184624218108643043e+2275802480651', 62, 1);
  t('-9.43461909417195191736082247633595980338e-44506', '3147834.7690127', '-2.997177357289012066606108455208457679374102048833411593840812447280717383143e-44512', 76, 3);
  t('8.0096127e-8067961309810', '-2.4861943641101754193763318966517820013802487677849556e+28737902', '-3.2216357721761189414886402877242567323592454396651199043193694782512026268261382719200753e-8067990047712', 89, 1);
  t('-3e+47914044916', '7.39032227496857769766000637292224305e+3151003', '-4.059362891603743292519367648173262662931001810631877144424048461001341365751815846570031160429e+47910893912', 94, 6);
  t('-8.794913236e-82269964025977', '1.99104078230641940199220131540233592723e-4032340415018', '-4.41724414394565158181607764560046776038373016888813781624061504013452734e-78237623610959', 73, 5);
  t('1.41968105004406037270658e-9756', '-6.45346706e+86374', '-2.19987339649341972887761202890528e-96131', 33, 5);
  t('2.42668305687527863637722979736247525202724198845375321e+21217079928', '-1e-32283604', '-2.42668305687527863637722979736247525202724198845375321e+21249363532', 75, 2);
  t('4.5960279e+6886974', '3.384439396e-9878', '1.3579879448962660639115193658500954289210738167403131127008072e+6896852', 62, 1);
  t('8.20302760909992674523110835076e-3190635561', '-0.00593408148019', '-1.38235843853584542319248668631e-3190635558', 30, 3);
  t('-1.446056788247471381740e+2539', '5.8746691249772632743e-204100', '-2.461511886855530870086019294051905971145867015553516078749595608714911097966096169396492e+206638', 88, 6);
  t('8.32020591493941974120926689522320401504559e+824', '-4.13656016816944653681263343833701120610642568125182e+36485506', '-2.0113827859e-36484682', 11, 5);
  t('-5.887058727145392610657250e-2197', '3.78194900860346014025947378458706e-8294267936', '-1.556620333524611679761108765636991252383460065276994885579337256044375049207119109e+8294265739', 82, 4);
  t('-7.6723323078147436832774113998379e+59830074', '5.39266e+7420616030062', '-1.422736146505573072153151023768956322111907667088227331224293762262037658595201625913743e-7420556199988', 88, 3);
  t('-2e+520902', '2.096256902873932e+850706884', '-9.540815332596088537412307660142977619618078320312637700439893846e-850185983', 64, 3);
  t('-5.91755623713028987336574603543834970e-772733', '-3.396837696660708e-743484', '1.742078004771201e-29249', 16, 4);
  t('-4.4261203993578076655836e+787568578554', '-1.09996983956065779587019620309266550937270720011352e+59408794921163', '4.0238561460245650861475347988199477648605072309975919369249233277670985289138655590183690383863e-58621226342609', 95, 1);
  t('-5.461625333592658715783389213939486766866733272971751e-8493558571126', '-7.04460022714407607499e-639442', '7.752924449208716784930185168011748e-8493557931685', 34, 4);
  t('1.615766164740165668562139888290618789895261e+863655', '-46413982727354151235644606048354659364.730096733328316079589', '-3.4812055975276420840108237250312e+863617', 32, 4);
  t('3.09253256167906696109040961441832872372394818118267e+73897720', '-2.0e-95602765', '-1.546266280839533480545204807209164361861974090591335e+169500485', 54, 5);
  t('-9.26915652715987334687941849664813e-6847', '8.76573855902163943304395e-34831559', '-1.05743018283611932638742e+34824712', 24, 4);
  t('0.428254', '6.08587970298546333861572751514e-437013115144', '7.036846288465372292244370006092052112772901167702223e+437013115142', 52, 1);
  t('7.690393e+80', '-7.604559818520535e-58509606603', '-1.0112870676972548059475609571154e+58509606683', 32, 1);
  t('-8.7801511725350736391998727179413695428856e-46359854365', '2.656781732693285651736344541155412284165970972485e-52097919051350', '-3.30480711474716594625752093810676249541657e+52051559196985', 42, 1);
  t('-1.27344496055459765937e+271188', '-8.32335945788509645225021195304781189636479e+3554126038', '1.52996511444451123611236024918713725931624703340785401601483423e-3553854851', 65, 4);
  t('-3.65362545705027531865430033750885949341477432262361470e+31405568', '2.255132753727929182126744495e+730457', '-1.620137639795491831777414566302950142374935254948e+30675111', 49, 0);
  t('3.13885105182978617280583072515316525825875033e+332826699874507', '-9.338570883214e-738313', '-3.361168524695618746504239051218405301858301969629363403151813526982628773e+332826700612819', 73, 5);
  t('7.916679560335212649839057098992815521111e+719168799789', '-9.3934455354681629e-26112', '-8.4278761509214954487660555758252e+719168825900', 32, 3);
  t('-7.4050648550717e+50457757', '-6.2446992145755164825827251414468888532118248382938e+41758863714', '1.185816097881514920689222170540626820579389457787832291775344955065933067404e-41708405957', 76, 4);
  t('-8.929984835073619447381098534882495124442070497803e+58', '-6.587552599082064547988352408462234967908817169753363929e+31478', '1.355584596974293397998024892163270225765679420243e-31420', 49, 0);
  t('-4.60e+3009732668', '-5.937435922938207873836812582485666474437571935215e+593', '7.7474520309831614048631736455137088e+3009732074', 35, 3);
  t('-2.115852508924661360352573897314e-83492', '-3.615253334615070879769043191528101492811972e+633', '5.852570520206556541343590796945702040066516674420075510945e-84126', 58, 3);
  t('-2671969484816629153.223031082473984381206', '-6.958622418800189e-6567179660', '3.839796620661208652769876288190432072799554268055501997751724529e+6567179677', 64, 3);
  t('-9.80911731960529794611006207152828152873100802686147229842508e+6602461063', '5.3006989098e+8373928', '-1.8505e+6594087135', 5, 5);
  t('-4e+4572149243', '6e-202556705641', '-6.66666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666667e+207128854883', 96, 0);
  t('-0.00277841246858389626011775371318910', '1.8029407e-8536873', '-1.5410448433406025279243813804797351349e+8536870', 38, 2);
  t('4.7366313774491542565994e+38590', '2.406574840920790361677323136428733366735190265835e-255', '1.96820447754570996174179187399056431789444073676092111382443598571041094824984995983407986690911147e+38845', 99, 3);
  t('-6.8793e-250', '5.6614239616418945369613535047947337585454417557391441e-22075572042', '-1.21511832475533308422685002589067048097909869932735928124654100237696191523886602e+22075571792', 81, 0);
  t('-7.9691e+5609', '4.7681728586005155851684044977835802380288408393935015223658e-824', '-1.67131105274966347e+6433', 18, 2);
  t('1.0163534771988257390384167798155678794959252153462087931e-7715', '9.857359701875963927124446349477232948130e-437', '1.031060555703777856579890730713937713166817989375566680949e-7279', 58, 6);
  t('-9.7119632239e+8914261', '-7.649129054596519809384883678662352318e+253', '1.26968222847068850083141642946053701890587434551e+8914008', 48, 1);
  t('3.82363000571632324625934315425582292e+9180', '-1.80015752902682364919840204196030449769953e-99958', '-2.12405300317e+109138', 12, 5);
  t('-9.98558786096530479519949931833477023441119520278653470635367e-9264430688990', '6.171492751155687501557785e-17356584223488', '-1.61801824349471711425087129591403125504623734104116821827862506496897452602816398117264777747324619e+8092153534498', 99, 6);
  t('-3.12121494951535774296813228065142594e-478824224292', '8.50191771196677938994457457569983125317092894e+3040', '-3.6711893189957913554e-478824227333', 20, 2);
  t('-3.905945148e-172570', '8.89926102560e-12918480850270', '-4.389066841352320025379703702395017431e+12918480677699', 37, 4);
  t('-5.836899e-182382713103', '6.38e+19389700249', '-9.14874451410658307210031347962382445141065830721003134796238244514106583073e-201772413353', 75, 3);
  t('-8.594943581149000619660143727800023854493343961823123e+67054', '7.72e+2366970549', '-1.113334660770595935189137788575132623639034191946000388601036269430051813471502590673575e-2366903495', 88, 5);
  t('4.422365172768345093341e-2474975', '8.142617173685931298235094554798454571552396073414454172878e+9459870092', '5.431134828565771941340664370852421405016220824937189375783467328373642722704174130143294674583e-9462345068', 94, 4);
  t('-7.510777732366610825653301072159051623394773695674094995e-3800345125', '-7.54354768818769002580340046770133074466784814214428947940e+5562', '9.9565589598215265433145400278831188933392065e-3800350688', 45, 0);
  t('6.1671423e+660', '6.873408012860653767381e+81013597436', '8.972466480181041945852762170983402582389595248407929170371790700151282177e-81013596777', 73, 5);
  t('2.02905811715297381814621675599006739725025165e-51', '8.306267856406390279349177689134436968649054e-88657834', '2.443e+88657782', 4, 2);
  t('-7.30e-114863902', '-6.58150691865777045498e+8689204889449', '1.10916847618975970006982958021532537105975629156199373295733575150771e-8689319753351', 69, 4);
  t('-16267894490157915816.761057622507335134', '-2.6263660e-57826799', '6.19406986313328599927087756333555000864312133e+57826817', 45, 1);
  t('4.4960822e+39834', '7.33025617737765e-5481077978', '6.1335949129249167141534547767142719606447193876923e+5481117811', 50, 2);
  t('11645961997440560519.2779', '7.61704147340133475340398370e-85', '1.52893509088905360245309042010759347198609170499739373773404875920893e+103', 69, 6);
  t('-3.200086468434878624724682260146719593251199021453006e+804185673862240', '-3.263005e+866674275403947', '9.80717611047142e-62488601541708', 15, 2);
  t('-1.195589034688348019231392777232e+387700847749', '1.2e+193718376381', '-9.9632419557e+193982471367', 11, 1);
  t('-7.7e+1775502', '261635432706832846.96155773903613920484498458287709539496460', '-2.9430264549175137e+1775485', 17, 1);
  t('0.00000546080760524465654739580663090511882514661697598240953776895', '-6.157849021385171979316104052444661102794551310655966818e+72472889810', '-8.86804399e-72472889817', 9, 2);
  t('2.8745074607815233067829206777233019531e-3845392237116', '-6.551146743e-662', '-4.3877928148273861780948366061e-3845392236455', 29, 5);
  t('-5.009861751554417108711752976e-74', '6.4649177022317048296284966530107e-89', '-7.749304758860272920489918385655728441338390550598260504191304956289874399458870815138e+14', 85, 5);
  t('6.6207966524387094348639971078e+8382394', '6.7558172830535640711220000733714793906928341502e-2246070', '9.80014167796760391792195206447498138446966736663067272302432667301486207936e+10628463', 75, 4);
  t('4.4513588705101755662795e+635486688804809', '-6.556877492876530213198836587854853e-4323337939', '-6.78883946717959097065518224297035382056172928446181486912391244568509234794436034526869106696e+635491012142747', 93, 4);
  t('-6.16322132e+994316265', '-4.3969e-2987200797', '1.4017196934203643476085423e+3981517062', 26, 1);
  t('-8.332e+2033', '-5.24183781e-7534', '1.5895188485429311671129328589432262498789522829589418372332279392e+9567', 65, 2);
});
