/* This file octo_double_gpufun.h defines the arithmetical operations 
   for octo double numbers on the GPU.

A octo double is defined by eight doubles.
The naming of the eight doubles of an octo double extends the naming
of the four doubles in a quad double (hihi, lohi, hilo, lolo) into
hihihi, lohihi, hilohi, lolohi, hihilo, lohilo, hilolo, lololo,
listed in decreasing order of significance.

The algorithms are from the CAMPARY and QD software libraries
with the modification that an octo double is not stored as an array
of eight doubles, but plainly by eight double numbers.

All functions have the prefix odg_ to avoid name clashes. */

#ifndef __octo_double_gpufun_h__
#define __octo_double_gpufun_h__

/************************** renormalizations **************************/

__device__ __forceinline__ void odg_renorm8
 ( double f0, double f1, double f2, double f3, double f4, double f5,
   double f6, double f7, double f8, double *pr, double *r0, double *r1,
   double *r2, double *r3, double *r4, double *r5, double *r6, double *r7 );
/*
 * DESCRIPTION :
 *   Definitions common to fast_renorm2L<9,8> and renorm2L_4Add1<8,8>
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   f0       most significant word;
 *   f1       second most significant word;
 *   f2       third most significant word;
 *   f3       fourth most significant word;
 *   f4       fifth most significant word;
 *   f5       sixth most significant word;
 *   f6       seventh most significant word;
 *   f7       least significant word;
 *   pr       computed by the start of the renormalization.
 *
 * ON RETURN :
 *   pr       updated value by renormalization;
 *   r0       highest part of a octo double number;
 *   r1       second highest part of an octo double number;
 *   r2       third highest part of an octo double number.
 *   r3       fourth highest part of an octo double number;
 *   r4       fourth lowest part of an octo double number;
 *   r5       third lowest part of an octo double number;
 *   r6       second lowest part of an octo double number;
 *   r7       lowest part of an octo double number. */

__device__ __forceinline__ void odg_fast_renorm
 ( double x0, double x1, double x2, double x3, double x4, double x5,
   double x6, double x7, double x8, double *r0, double *r1, double *r2,
   double *r3, double *r4, double *r5, double *r6, double *r7 );
/*
 * DESCRIPTION :
 *   The definition is based on the fast_renorm2L<9,8>,
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   x0       most significant word;
 *   x1       second most significant word;
 *   x2       third most significant word;
 *   x3       fourth most significant word;
 *   x4       fifth most significant word;
 *   x5       sixth most significant word;
 *   x6       seventh most significant word;
 *   x7       eighth most significant word;
 *   x8       least significant word.
 *
 * ON RETURN :
 *   r0       highest part of an octo double number;
 *   r1       second highest part of an octo double number;
 *   r2       third highest part of an octo double number.
 *   r3       fourth highest part of an octo double number;
 *   r4       fourth lowest part of an octo double number;
 *   r5       third lowest part of an octo double number;
 *   r6       second lowest part of an octo double number;
 *   r7       lowest part of an octo double number. */

__device__ __forceinline__ void odg_renorm_add1
 ( double x0, double x1, double x2, double x3, double x4, double x5,
   double x6, double x7, double y, double *r0, double *r1, double *r2,
   double *r3, double *r4, double *r5, double *r6, double *r7 );
/*
 * DESCRIPTION :
 *   The definition is based on the renorm2L_4Add1<8,8>,
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   x0       most significant word of an octo double x;
 *   x1       second most significant word of an octo double x;
 *   x2       third most significant word of an octo double x;
 *   x3       fourth most significant word of an octo double x;
 *   x4       fourth lowest part of an octo double number x;
 *   x5       third lowest part of an octo double number x;
 *   x6       second lowest part of an octo double number x;
 *   x7       least significant word of an octo double x;
 *   y        double to be added to x.
 *
 * ON RETURN :
 *   r0       highest part of x + y;
 *   r1       second highest part of x + y;
 *   r2       third highest part of x + y;
 *   r3       fourth highest part of x + y;
 *   r4       fourth lowest part of x + y;
 *   r5       third lowest part of x + y;
 *   r6       second lowest part of x + y;
 *   r7       lowest part of x + y. */

/************************ copy and abs *******************************/

__device__ __forceinline__ void odg_copy
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double *b_hihihi, double *b_lohihi, double *b_hilohi, double *b_lolohi,
   double *b_hihilo, double *b_lohilo, double *b_hilolo, double *b_lololo );
/*
 * DESCRIPTION :
 *   Copies the octo double a (a_hihihi, a_lohihi, a_hilohi, a_lolohi, 
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) to the octo double b (b_hihihi,
 *   b_lohihi, b_hilohi, b_lolohi, b_hihilo, b_lohilo, b_hilolo, b_lololo). */

__device__ __forceinline__ void odg_abs
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double *b_hihihi, double *b_lohihi, double *b_hilohi, double *b_lolohi,
   double *b_hihilo, double *b_lohilo, double *b_hilolo, double *b_lololo );
/*
 * DESCRIPTION :
 *   Returns in b the absolute value of a. */

/****************** additions and subtractions ************************/

__device__ __forceinline__ void odg_add
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b_hihihi, double b_lohihi, double b_hilohi, double b_lolohi,
   double b_hihilo, double b_lohilo, double b_hilolo, double b_lololo,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a + b.
 *   Adds two octo doubles a (a_hihihi, a_lohihi, a_hilohi, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) and b (b_hihihi, b_lohihi,
 *   b_hilohi, b_lolohi, b_hihilo, b_lohilo, b_hilolo, b_lololo)
 *   to make the octo double c (c_hihihi, c_lohihi, c_hilohi, c_lolohi,
 *   c_hihilo, c_lohilo, c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b;
 *   b_lololo is the lowest part of the octo double b.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a + b;
 *   c_lohihi is the second highest part of the octo double c = a + b;
 *   c_hilohi is the third highest part of the octo double c = a + b;
 *   c_lolohi is the fourth highest part of the octo double c = a + b;
 *   c_hihilo is the fourth lowest part of the octo double c = a + b;
 *   c_lohilo is the third lowest part of the octo double c = a + b;
 *   c_hilolo is the second lowest part of the octo double c = a + b;
 *   c_lololo is the lowest part of the octo double c = a + b. */

__device__ __forceinline__ void odg_inc
 ( double *a_hihihi, double *a_lohihi, double *a_hilohi, double *a_lolohi,
   double *a_hihilo, double *a_lohilo, double *a_hilolo, double *a_lololo,
   double b_hihihi, double b_lohihi, double b_hilohi, double b_lolohi,
   double b_hihilo, double b_lohilo, double b_hilolo, double b_lololo );
/*
 * DESCRIPTION : a = a + b.
 *   Adds to the octo double a (a_hihihi, a_lohihi, a_hilohi, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) the octo double b (b_hihihi,
 *   b_lohihi, b_hilohi, b_lolohi, b_hihilo, b_lohilo, b_hilolo, b_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b.
 *   b_lololo is the lowest part of the octo double b.
 *
 * ON RETURN :
 *   a_hihihi is the highest part of the octo double a + b;
 *   a_lohihi is the second highest part of the octo double a + b;
 *   a_hilohi is the third highest part of the octo double a + b;
 *   a_lolohi is the fourth highest part of the octo double a + b;
 *   a_hihilo is the fourth lowest part of the octo double a + b;
 *   a_lohilo is the third lowest part of the octo double a + b;
 *   a_hilolo is the second lowest part of the octo double a + b;
 *   a_lololo is the lowest part of the octo double a + b. */

__device__ __forceinline__ void odg_inc_d
 ( double *a_hihihi, double *a_lohihi, double *a_hilohi, double *a_lolohi,
   double *a_hihilo, double *a_lohilo, double *a_hilolo, double *a_lololo,
   double b );
/*
 * DESCRIPTION : a = a + b.
 *   Adds the double b to the octo double a (a_hihihi, a_lohihi, a_hilohi,
 *   a_lolohi, a_hihilo, a_lohilo, a_hilolo, a_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b        some double.
 *
 * ON RETURN :
 *   a_hihihi is the highest part of the octo double a + b;
 *   a_lohihi is the second highest part of the octo double a + b;
 *   a_hilohi is the third highest part of the octo double a + b;
 *   a_lolohi is the fourth highest part of the octo double a + b;
 *   a_hihilo is the fourth lowest part of the octo double a + b;
 *   a_lohilo is the third lowest part of the octo double a + b;
 *   a_hilolo is the second lowest part of the octo double a + b;
 *   a_lololo is the lowest part of the octo double a + b; */

__device__ __forceinline__ void odg_minus
 ( double *a_hihihi, double *a_lohihi, double *a_hilohi, double *a_lolohi,
   double *a_hihilo, double *a_lohilo, double *a_hilolo, double *a_lololo );
/*
 * DESCRIPTION :
 *   Flips the sign of a (a_hihihi, a_lohihi, a_hilohi, a_lolohi,
 *                        a_hihilo, a_lohilo, a_hilolo, a_lololo). */

__device__ __forceinline__ void odg_sub
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b_hihihi, double b_lohihi, double b_hilohi, double b_lolohi,
   double b_hihilo, double b_lohilo, double b_hilolo, double b_lololo,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a - b.
 *   Subtracts the octo double b (b_hihihi, b_lohihi, b_hilohi, b_lolohi,
 *   b_hihilo, b_lohilo, b_hilolo, b_lololo) from the octo double
 *   in a (a_hihihi, a_lohihi, a_hilohi, a_lolohi, a_hihilo, a_lohilo,
 *   a_hilolo, a_lololo) and places the result in the octo double
 *   c (c_hihihi, c_lohihi, c_hilohi, c_lolohi, c_hihilo, c_lohilo,
 *   c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b.
 *   b_lololo is the lowest part of the octo double b.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a - b;
 *   c_lohihi is the second highest part of the octo double c = a - b;
 *   c_hilohi is the third highest part of the octo double c = a - b;
 *   c_lolohi is the fourth highest part of the octo double c = a - b;
 *   c_hihilo is the fourth lowest part of the octo double c = a - b;
 *   c_lohilo is the third lowest part of the octo double c = a - b;
 *   c_hilolo is the second lowest part of the octo double c = a - b;
 *   c_lololo is the lowest part of the octo double c = a - b. */

/***************** multiplications and division ********************/

__device__ __forceinline__ void odg_mul_pwr2
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a * b, where b is a power of two.
 *   As b is a power of two, all components of the octo double
 *   are plainly multiplied by b.
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b        a power of two.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a * b;
 *   c_lohihi is the second highest part of the octo double c = a * b;
 *   c_hilohi is the third highest part of the octo double c = a * b;
 *   c_lolohi is the fourth highest part of the octo double c = a * b;
 *   c_hihilo is the fourth lowest part of the octo double c = a * b;
 *   c_lohilo is the third lowest part of the octo double c = a * b;
 *   c_hilolo is the second lowest part of the octo double c = a * b;
 *   c_lololo is the lowest part of the octo double c = a * b. */

__device__ __forceinline__ void odg_mul
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b_hihihi, double b_lohihi, double b_hilohi, double b_lolohi,
   double b_hihilo, double b_lohilo, double b_hilolo, double b_lololo,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a * b.
 *   Multiplies two octo doubles a (a_hihihi, a_lohihi, a_hilohi, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) and b (b_hihihi, b_lohihi,
 *   b_hilohi, b_lolohi, b_hihilo, b_lohilo, b_hilolo, b_lololo)
 *   to make the octo double c (c_hihihi, c_lohihi, c_hilohi, c_lolohi,
 *   c_hihilo, c_lohilo, c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b;
 *   b_lololo is the lowest part of the octo double b.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a * b;
 *   c_lohihi is the second highest part of the octo double c = a * b;
 *   c_hilohi is the third highest part of the octo double c = a * b;
 *   c_lolohi is the fourth highest part of the octo double c = a * b;
 *   c_hihilo is the fourth lowest part of the octo double c = a * b;
 *   c_lohilo is the third lowest part of the octo double c = a * b;
 *   c_hilolo is the second lowest part of the octo double c = a * b;
 *   c_lololo is the lowest part of the octo double c = a * b. */

__device__ __forceinline__ void odg_sqr
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a * a.
 *   Multiplies the octo double a (a_hihihi, a_lohihi, a_hilohi, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) with a
 *   to make the octo double c (c_hihihi, c_lohihi, c_hilohi, c_lolohi,
 *   c_hihilo, c_lohilo, c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a. 
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a * a;
 *   c_lohihi is the second highest part of the octo double c = a * a;
 *   c_hilohi is the third highest part of the octo double c = a * a;
 *   c_lolohi is the fourth highest part of the octo double c = a * a;
 *   c_hihilo is the fourth lowest part of the octo double c = a * a;
 *   c_lohilo is the third lowest part of the octo double c = a * a;
 *   c_hilolo is the second lowest part of the octo double c = a * a;
 *   c_lololo is the lowest part of the octo double c = a * a. */

__device__ __forceinline__ void odg_mul_od_d
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a * b.
 *   Multiplies the octo double a (a_hihihi, a_lohihi, a_hilolo, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) with b
 *   to make the octo double c (c_hihihi, c_lohihi, c_hilohi, c_lolohi,
 *   c_hihilo, c_lohilo, c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b        is some double.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a * b;
 *   c_lohihi is the second highest part of the octo double c = a * b;
 *   c_hilohi is the third highest part of the octo double c = a * b;
 *   c_lolohi is the fourth highest part of the octo double c = a * b;
 *   c_hihilo is the fourth lowest part of the octo double c = a * b;
 *   c_lohilo is the third lowest part of the octo double c = a * b;
 *   c_hilolo is the second lowest part of the octo double c = a * b;
 *   c_lololo is the lowest part of the octo double c = a * b. */

__device__ __forceinline__ void odg_div
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double b_hihihi, double b_lohihi, double b_hilohi, double b_lolohi,
   double b_hihilo, double b_lohilo, double b_hilolo, double b_lololo,
   double *c_hihihi, double *c_lohihi, double *c_hilohi, double *c_lolohi,
   double *c_hihilo, double *c_lohilo, double *c_hilolo, double *c_lololo );
/*
 * DESCRIPTION : c = a / b.
 *   Divides the octo double a (a_hihihi, a_lohihi, a_hilolo, a_lolohi,
 *   a_hihilo, a_lohilo, a_hilolo, a_lololo) by b (b_hihihi, b_lohihi,
 *   b_hilohi, b_lolohi, b_hihilo, b_lohilo, b_hilolo, b_lololo)
 *   to make the octo double c (c_hihihi, c_lohihi, c_lohihi, c_lolohi,
 *   c_hihilo, c_lohilo, c_hilolo, c_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a;
 *   a_hihilo is the fourth lowest part of the octo double a;
 *   a_lohilo is the third lowest part of the octo double a;
 *   a_hilolo is the second lowest part of the octo double a;
 *   a_lololo is the lowest part of the octo double a;
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b;
 *   b_lololo is the lowest part of the octo double b.
 *
 * ON RETURN :
 *   c_hihihi is the highest part of the octo double c = a / b;
 *   c_lohihi is the second highest part of the octo double c = a / b;
 *   c_hilohi is the third highest part of the octo double c = a / b;
 *   c_lolohi is the fourth highest part of the octo double c = a / b;
 *   c_hihilo is the fourth lowest part of the octo double c = a / b;
 *   c_lohilo is the third lowest part of the octo double c = a / b;
 *   c_hilolo is the second lowest part of the octo double c = a / b;
 *   c_lololo is the lowest part of the octo double c = a / b. */

/***************************** square root *****************************/

__device__ __forceinline__ void odg_sqrt
 ( double a_hihihi, double a_lohihi, double a_hilohi, double a_lolohi,
   double a_hihilo, double a_lohilo, double a_hilolo, double a_lololo,
   double *b_hihihi, double *b_lohihi, double *b_hilohi, double *b_lolohi,
   double *b_hihilo, double *b_lohilo, double *b_hilolo, double *b_lololo );
/*
 * DESCRIPTION :
 *   Returns in the octo double b (b_hihihi, b_lohihi, b_hilohi, b_lolohi,
 *   b_hihilo, b_lohilo, b_hilolo, b_lololo) the square root of the octo
 *   double a (a_hihihi, a_lohihi, a_hilohi, a_lolohi, a_hihilo, a_lohilo,
 *   a_hilolo, a_lololo).
 *
 * ON ENTRY :
 *   a_hihihi is the highest part of the octo double a;
 *   a_lohihi is the second highest part of the octo double a;
 *   a_hilohi is the third highest part of the octo double a;
 *   a_lolohi is the fourth highest part of the octo double a.
 *   a_hihilo is the fourth lowest part of the octo double a.
 *   a_lohilo is the third lowest part of the octo double a.
 *   a_hilolo is the second lowest part of the octo double a.
 *   a_lololo is the lowest part of the octo double a.
 *
 * ON RETURN :
 *   b_hihihi is the highest part of the octo double b;
 *   b_lohihi is the second highest part of the octo double b;
 *   b_hilohi is the third highest part of the octo double b;
 *   b_lolohi is the fourth highest part of the octo double b;
 *   b_hihilo is the fourth lowest part of the octo double b;
 *   b_lohilo is the third lowest part of the octo double b;
 *   b_hilolo is the second lowest part of the octo double b;
 *   b_lololo is the lowest part of the octo double b. */

#endif
