/*
 * Functions dealing with physical node coordinates in terms of
 * - cabinet / row / chassis / slot / node
 * - Gemini chip / SeaStar node.
 *
 * NID convention:
 * <cabinet>-<row><chassis><slot><node>
 * - each cabinet can accommodate 3 chassis (c1..c3)
 * - each chassis has 8 slots               (s0..s7)
 * - each slot contains 2 or 4 nodes        (n0..n3)
 *   o either 2 service nodes (n0/n3)
 *   o or 4 compute nodes     (n0..n3)
 *   o or 2 gemini chips      (g0/g1 serving n0..n3)
 *
 * Example: c0-0c1s0n1
 * - c0- = cabinet 0 (e.g. 9 cabinets, 0-8, here)
 * - 0   = row     0 (e.g. 2 rows, 0-1, here)
 * - c1  = chassis 1 (3 chassis per cabinet, 0-2)
 * - s0  = slot    0 (8 slots/chassis, 0-7)
 * - n1  = node    1 (4 nodes, 0-3)
 *
 * Example: c1-1c1s0g0
 * - Gemini chip 0 of
 * - slot 0 in cage 1 of cabinet 1, in row 1.
 * - The chip serves c1-1c1s0n0 and c1-1c1s0n1.
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"
#include "basil_torus.h"

/** Stringify node coordinates of @nc */
const char *nodecoord2string(const struct nodecoord *nc)
{
	static char nodestr[32];

	snprintf(nodestr, sizeof(nodestr), "c%u-%uc%us%un%u",
		 nc->cab, nc->row, nc->cage, nc->slot, nc->node);

	return nodestr;
}

/** Stringify Gemini coordinates of @nc */
const char *nodecoord2gemini_string(const struct nodecoord *nc)
{
	static char geministr[32];

	snprintf(geministr, sizeof(geministr), "c%u-%uc%us%ug%u",
		 nc->cab, nc->row, nc->cage, nc->slot, nc->node > 1);
	return geministr;
}

/** Break down node string @str into coordinates @nc */
bool string2nodecoord(const char *str, struct nodecoord *nc)
{
	unsigned int row, cab, cage, slot, nid;

	if (str == NULL)
		return false;
	if (sscanf(str, "c%u-%uc%us%un%u", &cab, &row, &cage, &slot, &nid) != 5)
		return false;
	if (cage > 2 || slot > 7 || nid > 3)
		return false;
	nc->cab  = cab;
	nc->row  = row;
	nc->cage = cage;
	nc->slot = slot;
	nc->node = nid;
	return true;
}

/** Convert @nodestr of type c#-#c#s#n# into Gemini string of type c#-#c#s#g# */
const char *node_string2gemini_string(const char *nodestr)
{
	struct nodecoord nc;

	if (!string2nodecoord(nodestr, &nc))
		return NULL;
	return nodecoord2gemini_string(&nc);
}

/** Validate node coordinate @nc against @ti. */
bool is_valid_nodecoord(const struct nodecoord *nc, const struct torus_info *ti)
{
	return	nc->cab <= ti->cabs - 1 && nc->row <= ti->rows - 1 &&
		nc->cage <= 2 && nc->slot <= 7 && nc->node <= 3;
}

/** Verbosely interpret node coordinates */
void interpret_nodecoord(const struct nodecoord *nc)
{
	printf("%s: cabinet %u in row %u, node %u of slot %u in chassis %u\n",
		nodecoord2string(nc), nc->cab, nc->row,
		nc->node, nc->slot, nc->cage);
}

void interpret_nodecoord_str(const char *str)
{
	struct nodecoord nc;

	if (! string2nodecoord(str, &nc))
		printf("Invalid node-coordinate string '%s'\n", str);
	else
		interpret_nodecoord(&nc);
}

/*
 * Mapping numerical node ID into cpu/slot/cage/cabinet/row
 */

/* Base-2 logarithm of val > 0 */
static uint8_t lb(uint64_t val)
{
	uint8_t r = 0;

	while (val >>= 1)
		r++;
	return r;
}

/* Return minimum number of bits needed to represent @val */
static uint8_t num_bits(uint64_t val)
{
	return lb(val) + 1;
}

/* Return bitmask of up to 64 bits */
static uint64_t get_mask(uint64_t val)
{
	return val == 0 ? 0 : ~0ULL >> (64 - num_bits(val));
}


/**
 * nid2node - break node ID down into node coordinates
 * @nid:		numerical node ID
 * @num_rows_in_system:	the total count of rows the system has
 *
 * Returns statically allocated result.
 */
struct nodecoord *nid2node(const uint32_t nid, struct torus_info *ti)
{
	static struct nodecoord nc;

	switch (ti->network) {
	case ASIC_XT_SEASTAR:
		nc.cab  = nid >> (7 + lb(ti->rows));
		if (ti->rows < 2)
			nc.row = 0;
		else
			nc.row = (nid >> 7) & get_mask(ti->rows - 1);
		nc.cage = (nid >> 5) & 0x3;
		nc.slot = (nid >> 2) & 0x7;
		nc.node = nid & 0x3;
		break;
	case ASIC_XE_GEMINI:
		/* XXX FIXME: so far only tested on 2-cabinet, single-row system */
		/* XXX FIXME: guesswork, lacking rule to generalize things below */
		/* XXX FIXME: cabinet number irrespective of row numbering */
		nc.cab  = nid / XE_NODES_PER_CABINET;

		nc.row  = 0;	/* XXX FIXME */

		nc.cage = (nid % XE_NODES_PER_CABINET) >> 5;
		/*
		 * XXX FIXME: Hypothesis is that
		 *	- even-numbered cabinets count cages bottom to top,
		 *	- odd-numbered cabinets count cages top to bottom.
		 */
		if (nc.cab & 1)
			nc.cage = 2 - nc.cage;

		nc.slot = (nid & 0x10 ? ~nid : nid) >> 1 & 0x7;

		/* XXX FIXME: Hypothesis is based on continuous cage numbering,
		 * computed as floor(nid/32):
		 *	- if that number is even, use node number
		 *	- if that number is odd, flip the ordering
		 */
		nc.node = ((nid & 0x10) >> 3) | (nid & 1);
		if ((nid >> 5) & 1)
			nc.node = 2 ^ nc.node;
		break;
	}
	return &nc;
}

/** Convert node coordinates into torus information */
struct torus_coord nid2toruscoord(const struct nodecoord *nc,
				  const struct torus_info *ti)
{
	struct torus_coord tc;

	if (ti->cabling == CABLING_UNKNOWN)
		fatal("unknown torus type");

	switch (ti->network) {
	case ASIC_XE_GEMINI:
		tc = nid2gemini_toruscoord(nc, ti);
		break;
	case ASIC_XT_SEASTAR:
		tc = nid2seastar_toruscoord(nc, ti);
		/* X is 0 on single-chassis systems (e.g. gele, fred) */
		if (tc.x && tc.x >= ti->x_max)
			fatal("invalid x = %d (max %d)", tc.x, ti->x_max);
		if (tc.y >= ti->y_max)
			fatal("invalid y = %d (max %d)", tc.y, ti->y_max);
		if (tc.z >= ti->z_max)
			fatal("invalid z = %d (max %d)", tc.z, ti->z_max);
		break;
	}
	return tc;
}
