#pragma once
#include "Thread.h"
#include "Core/Gen/CppTypes.h"
#include "Core/Lock.h"
#include "Template.h"

namespace storm {
	STORM_PKG(core.lang);

	class Named;
	class NameSet;
	class World;

	/**
	 * Represents all instantiations of a template class available to C++.
	 *
	 * Note: This object needs to be thread-safe in the find() function as that is called from any thread of C++.
	 */
	class TemplateList : public ObjectOn<Compiler> {
		STORM_CLASS;
	public:
		// Create.
		TemplateList(World *world, TemplateCppFn *t);

		// Add everything in here to NameSet.
		void addTo(NameSet *to);

		// Get what to add into.
		inline NameSet *addTo() const { return addedTo; }

		// Find an instantiaton. Generates it if neccessary.
		Type *find(Nat *elems, Nat count);

		// Run 'fn' for all named objects in here.
		typedef void (*NamedFn)(Named *);
		void forNamed(NamedFn fn);

	private:
		// Template we're representing.
		TemplateCppFn *templ;

		// World in which to lookup type indices.
		UNKNOWN(PTR_NOGC) World *world;

		// Lock used for 'find'.
		Lock *lock;

		// A node in the list.
		struct Node;

		// Multi-level list of generated templates.
		UNKNOWN(PTR_GC) Node *root;

		// Added to a NameSet? If non-null, we shall add any newly-generated templates here.
		NameSet *addedTo;

		// Find a template specialization inside the structure.
		Type *findAt(Nat *elems, Nat count, Node *node, Nat at);

		// Insert a type.
		void insertAt(Nat *elems, Nat count, Type *insert, Node *&node, Nat at);

		// Allocate a node of a specific size.
		Node *allocNode(Nat count);
		Node *allocNode(const Node *original, Nat newCount);

		// Generate a new template.
		Type *generate(Nat *elems, Nat count);

		// For each.
		void forNamed(Node *at, NamedFn fn);

		// Add to a NameSet.
		void addTo(Node *at, NameSet *to);
	};

}
