/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the XMPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * XMPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: xmpi_run.cc,v 1.13 2003/09/03 04:28:54 jsquyres Exp $
 *
 *	Function:	- application specification panel in
 *			  the builder dialog
 */

#include <fstream>
#include <string>

using namespace std;

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <Xm/Xm.h>
#include <Xm/ArrowBG.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/SelectioB.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>
#include <Xm/Xm.h>
#include <Xm/FileSB.h>
#include <Xm/ToggleBG.h>

#include "xmpi_all_list.h"
#include "app_mgmt.h"
#include "lam.h"
#include "xmpi.h"
#include "xmpi_run.h"
#include "xmpi_copies.h"
#include "xmpi_misc.h"
#include "xmpi_error.h"
#include "xmpi_ctl.h"
#include "xmpi_getline.h"
#include "xmpi_view.h"
#include "xmpi_browse.h"
#include "xmpi_nodes.h"
#include "xmpi_matrix.h"
#include "xmpi_kiviat.h"
#include "xmpi_trace.h"
#include "xmpi_vendor.h"
#include "xmpi_sys.h"

/*
 * private functions
 */
static void run_run_cb(Widget);
static int run_save(char *);
static void run_change_cb(Widget, XtPointer, XtPointer);
static void run_cancel_cb();
static void run_save_cb(Widget);
static void run_save_ok_cb(Widget, XtPointer, 
			   XmSelectionBoxCallbackStruct*);
static void browse_prog_cb(Widget, XtPointer, 
			   XmFileSelectionBoxCallbackStruct*);
static void run_load_cb(Widget, XtPointer, XtPointer);
static void load_prog_cb(Widget, XtPointer,
			 XmFileSelectionBoxCallbackStruct*);
static void run_new_cb(Widget, XtPointer, XtPointer);
static void run_copy_cb(Widget, XtPointer, XtPointer);
static void run_delete_cb(Widget, XtPointer, XtPointer);
static void run_select_cb(Widget, XtPointer, XmListCallbackStruct*);
void xmpi_run_file();

// to enable/disable the input btns
static void run_enable_btns();
static void run_disable_btns();
// helper to update the status bar
static void run_status_change(char*);
// update the left-most pane
static void run_update(char*, int, char*);


/*
 * public variables
 */
int xmpi_do_rerun = 0;		       /* re-running application? */
int xmpi_app_nprocs = 0;	       /* number of procs in appl. */
struct _gps *xmpi_app_procs = 0;       /* appl. GPS array */
Widget xmpi_run_w = 0;		       /* run dialog shell */

// Stuff for the vendor pane
struct xmpi_menu_item *run_options;
int run_options_length;
Widget *run_options_widgets = 0;

struct xmpi_menu_item *schema_options;
int schema_options_length;
Widget *schema_options_widgets = 0;

struct xmpi_node *nodes;
int nodes_length;

extern int xmpi_trinit;

/*
 * private variables
 */
static char *custom = 0;	       /* custom aschema filename */
static char prog_dir[XMPI_PATHMAX] = "";	/* dir. selected prog. */
static char tempfile[32];
static Widget args_w;
static Widget fullpath_w;
static Widget prog_w;
static Widget copies_w;
static Widget list_w;
static Widget xmpi_run_wrap_w;         // Frame widget to sit in xmpi_run_w

static Widget btn_new_w;
static Widget btn_copy_w;
static Widget btn_delete_w;
static Widget btn_load_w;
static Widget btn_save_w;
static Widget btn_run_w;
static Widget btn_cancel_w;

static Widget status_w;

static char delete_trans[] = "#override\n\
	<Key>osfDelete: xmpi_run_delete()\n\
	<Key>osfBackSpace: xmpi_run_delete()";

static Widget fdialog = 0;	       /* file selection dialog */
static Widget load_fdialog = 0;

static bool update = true;

/*
 *	xmpi_run_dialog
 *
 *	Function:	- creates widgets for mpirun dialog window
 *			- pops up dialog
 *	Accepts:	- parent widget
 *			- initial boot schema
 */
void
xmpi_run_dialog(Widget parent_w)
{
  Widget dialog_mgr_w;  	       /* manager for three horiz panes */
  Widget status_frame_w;               // frame for the status label
  Widget mgr_w;			       /* prog/app pane manager (left pane) */
  Widget prog_mgr_frame_w;             // outline for program area mgr
  Widget prog_mgr_w;		       // program area manager 
  Widget current_app_lbl_w;
  Widget list_action_mgr_w;            // buttons to control prog list
  Widget action_mgr_w;		       /* action area manager */
  Widget w;                            // generic list...
  Dimension width;		       /* alignment of labels */
  Dimension height;
  XmString xstr;
  Arg args[1];

  int numb_horz_panes = 0;  // needed because we might only have 2...
  if (xmpi_run_w) {

    /*
     * Dialog window already created, just pop it on the screen.
     */
    XtMapWidget(xmpi_wmshell(xmpi_run_w));
    XtManageChild(xmpi_run_wrap_w);
    XtPopup(xmpi_run_w, XtGrabNone);
    return;
  }

  if (getcwd(prog_dir, XMPI_PATHMAX) == NULL)
    strcpy(prog_dir, "");

  // start by getting the vendor information.  This determines the
  // size window about to be drawn.
  if (xmpi_sys_get_vendor_info(&run_options, &run_options_length,
			       &schema_options, &schema_options_length,
			       &nodes, &nodes_length) != 0) {
    // we hit an error.  Hopefully, the vendor threw an error message
    return;
  }
  
  // if we received any menu items, need to make sure to create space.
  if ((run_options_length == 0) && (schema_options_length == 0))
    numb_horz_panes = 2;
  else
    numb_horz_panes = 3;

  /*
   * Create the dialog window.
   */
  xmpi_run_w = XtVaCreatePopupShell("buildrun_pop",
			      xmDialogShellWidgetClass, xmpi_wmshell(parent_w),
			      XmNdeleteResponse, XmUNMAP,
			      XmNtitle, "XMPI Application Builder",
			      XmNminHeight, 300,
			      NULL);

  xmpi_run_wrap_w = XtVaCreateWidget("xmpi_run_wrap_w",
				     xmFormWidgetClass, xmpi_run_w,
				     XmNmaxWidth, 50,
				     NULL);

  // The status bar on the bottom.  Frame provides the depth
  status_frame_w = XtVaCreateManagedWidget("status_frame_w",
					   xmFormWidgetClass, xmpi_run_wrap_w,
					   XmNmarginHeight, 1,
					   XmNmarginWidth, 1,
					   XmNshadowThickness, 5,
					   XmNshadowType, XmSHADOW_ETCHED_IN,
					   XmNrightAttachment, XmATTACH_FORM,
					   XmNleftAttachment, XmATTACH_FORM,
					   XmNbottomAttachment, XmATTACH_FORM,
					   NULL);

  status_w = XtVaCreateManagedWidget("status",
				     xmLabelWidgetClass, status_frame_w,
				     XmNtopAttachment, XmATTACH_FORM,
				     XmNrightAttachment, XmATTACH_FORM,
				     XmNleftAttachment, XmATTACH_FORM,
				     XmNbottomAttachment, XmATTACH_FORM,
				     XmNalignment, XmALIGNMENT_BEGINNING,
				     NULL);

  // jump through hopes to get the label to be the right size...
  XtVaGetValues(status_w, XmNheight, &height, NULL);
  xstr = XmStringCreateSimple((char*) "");
  XtVaSetValues(status_w, XmNlabelString, xstr, NULL);
  XmStringFree(xstr);
  XtVaSetValues(status_w, 
		XmNheight, height, 
		XmNrecomputeSize, False,
		NULL);

  // The three-paned horizontal frame holding everything but the
  // status bar
  dialog_mgr_w = XtVaCreateWidget("build_run_mgr",
				  xmFormWidgetClass, xmpi_run_wrap_w,
				  XmNfractionBase, numb_horz_panes,
				  XmNtopAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_FORM,
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNbottomAttachment, XmATTACH_WIDGET,
				  XmNbottomWidget, status_frame_w,
				  NULL);

  //
  //
  // LEFT MOST PANE: PROGRAM NAME/ARGS AND SCHEMA CONTROLS / LIST
  //
  //
  mgr_w = XtVaCreateWidget("app_mgr",
			   xmPanedWindowWidgetClass, dialog_mgr_w,
			   XmNsashWidth, 1,
			   XmNsashHeight, 1,
			   XmNseparatorOn, False,
			   XmNleftAttachment, XmATTACH_FORM,
			   XmNrightAttachment, XmATTACH_POSITION,
			   XmNrightPosition, 1,
			   XmNrightOffset, 5,
			   XmNleftOffset, 1,
			   XmNtopAttachment, XmATTACH_FORM,
			   XmNbottomAttachment, XmATTACH_FORM,
			   NULL);

  prog_mgr_frame_w = XtVaCreateWidget("prog_frame",
				      xmFrameWidgetClass, mgr_w,
				      XmNmarginHeight, 3,
				      XmNmarginWidth, 3,
				      XmNshadowThickness, 2,
				      NULL);

  prog_mgr_w = XtVaCreateWidget("prog_mgr",
				xmFormWidgetClass, prog_mgr_frame_w,
				XmNfractionBase, 4,
				XmNleftAttachment, XmATTACH_FORM,
				XmNrightAttachment, XmATTACH_FORM,
				XmNtopAttachment, XmATTACH_FORM,
				XmNbottomAttachment, XmATTACH_FORM,
				NULL);
  
  // Build the number of copies entry area.
  // This is out of order so that we can get the width of the
  // widest line (which is copy)
  w = XtVaCreateManagedWidget("Copy:",
			      xmLabelGadgetClass, prog_mgr_w,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 2,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 3,
			      XmNleftAttachment, XmATTACH_FORM,
			      NULL);
  copies_w = xmpi_copies_create(prog_mgr_w);
  XtVaSetValues(copies_w,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 2,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 3,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
  XtManageChild(copies_w);
  XtVaGetValues(w, XmNwidth, &width, NULL);

  /*
   * Build the program name entry area.
   */  

  w = XtVaCreateManagedWidget("Prog:",
			      xmLabelGadgetClass, prog_mgr_w,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 1,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNwidth, width,
			      NULL);

  Widget browse_w;
  browse_w = XtVaCreateManagedWidget("Browse",
				     xmPushButtonGadgetClass, prog_mgr_w,
				     XmNdefaultButtonShadowThickness, 0,
				     XmNrightAttachment, XmATTACH_FORM,
				     XmNtopAttachment, XmATTACH_POSITION,
				     XmNtopPosition, 0,
				     XmNbottomAttachment, XmATTACH_POSITION,
				     XmNbottomPosition, 1,
				     NULL);

  prog_w = XtVaCreateManagedWidget("prog_text",
				   xmTextFieldWidgetClass, prog_mgr_w,
				   XmNleftAttachment, XmATTACH_WIDGET,
				   XmNleftWidget, w,
				   XmNrightAttachment, XmATTACH_WIDGET,
				   XmNrightWidget, browse_w,
				   XmNtopAttachment, XmATTACH_POSITION,
				   XmNtopPosition, 0,
				   XmNbottomAttachment, XmATTACH_POSITION,
				   XmNbottomPosition, 1,
				   NULL);

  XtAddCallback(prog_w, XmNvalueChangedCallback, run_change_cb, xmpi_run_w);

  XtAddCallback(browse_w, XmNactivateCallback, 
		(XtCallbackProc) xmpi_run_file , xmpi_run_w);

  /*
   * Build the program arguments entry area.
   */
  w = XtVaCreateManagedWidget("Args:",
			      xmLabelGadgetClass, prog_mgr_w,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 1,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 2,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNwidth, width,
			      NULL);
  args_w = XtVaCreateManagedWidget("args_text",
				   xmTextFieldWidgetClass, prog_mgr_w,
				   XmNtopAttachment, XmATTACH_POSITION,
				   XmNtopPosition, 1,
				   XmNbottomAttachment, XmATTACH_POSITION,
				   XmNbottomPosition, 2,
				   XmNleftAttachment, XmATTACH_WIDGET,
				   XmNleftWidget, w,
				   XmNrightAttachment, XmATTACH_FORM,
				   NULL);
  XtAddCallback(args_w, XmNvalueChangedCallback, run_change_cb, xmpi_run_w);

  /*
   * Build the "Use Full Pathname" toggle button.
   */
  xstr = XmStringCreateSimple((char*) "Use Full Pathname");
  fullpath_w = XtVaCreateManagedWidget("fullpath_toggle",
				       xmToggleButtonWidgetClass, prog_mgr_w,
				       XmNalignment, XmALIGNMENT_BEGINNING,
				       XmNnavigationType, XmTAB_GROUP,
				       XmNtopAttachment, XmATTACH_POSITION,
				       XmNtopPosition, 3,
				  XmNbottomAttachment, XmATTACH_POSITION,
				       XmNbottomPosition, 4,
				       XmNleftAttachment, XmATTACH_FORM,
				       XmNlabelString, xstr,
				       NULL);
  XtAddCallback(fullpath_w, XmNvalueChangedCallback, 
		run_change_cb, xmpi_run_w);

  XmStringFree(xstr);

  XtManageChild(prog_mgr_w);
  XtManageChild(prog_mgr_frame_w);

  xstr = XmStringCreateSimple((char*) "Current Application");
  current_app_lbl_w = XtVaCreateManagedWidget("banner",
				       xmLabelWidgetClass, mgr_w,
				       XmNlabelString, xstr,
				       NULL);
  XmStringFree(xstr);

  // add row for copy, new, delete
  list_action_mgr_w = XtVaCreateManagedWidget("action_mgr_schema",
				       xmFormWidgetClass, mgr_w,
				       XmNfractionBase, 3,
				       NULL);

  btn_new_w = XtVaCreateManagedWidget("New", 
			      xmPushButtonGadgetClass, list_action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 3,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNrightAttachment, XmATTACH_POSITION,
			      XmNrightPosition, 1,
			      NULL);
  
  XtAddCallback(btn_new_w, XmNactivateCallback, run_new_cb, NULL);

  btn_copy_w = XtVaCreateManagedWidget("Copy", 
			      xmPushButtonGadgetClass, list_action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 3,
			      XmNleftAttachment, XmATTACH_POSITION,
			      XmNleftPosition, 1,
			      XmNrightAttachment, XmATTACH_POSITION,
			      XmNrightPosition, 2,
			      NULL);
  
  XtAddCallback(btn_copy_w, XmNactivateCallback, run_copy_cb, NULL);

  btn_delete_w = XtVaCreateManagedWidget("Delete", 
			      xmPushButtonGadgetClass, list_action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 3,
			      XmNleftAttachment, XmATTACH_POSITION,
			      XmNleftPosition, 2,
			      XmNrightAttachment, XmATTACH_FORM,
			      NULL);
  XtAddCallback(btn_delete_w, XmNactivateCallback, run_delete_cb, NULL);

  /*
   * Build the application scrolled list.
   */
  XtSetArg(args[0], XmNlistSizePolicy, XmCONSTANT);
  list_w = XmCreateScrolledList(mgr_w, (char*) "app_list", args, 1);
  XtVaSetValues(list_w,
		XmNselectionPolicy, XmSINGLE_SELECT,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		NULL);

  XtOverrideTranslations(list_w, XtParseTranslationTable(delete_trans));
  
  XtAddCallback(list_w, XmNsingleSelectionCallback,
		(XtCallbackProc) run_select_cb, NULL);

  // Add the <EMPTY> line, and select.
  xstr = XmStringCreateSimple((char*) "<EMPTY>");
  XmListAddItem(list_w, xstr, 0);
  XmListSelectItem(list_w, xstr, False);
  XmStringFree(xstr);

  // Create the dialog action area.
  // action_mgr_schema is the schema control row
  // action_mgr_program is the program (run/cancel) row
  action_mgr_w = XtVaCreateManagedWidget("action_mgr_schema",
				  xmFormWidgetClass, mgr_w,
				  XmNfractionBase, 2,
				  NULL);

  // Create the "Load" button.
  btn_load_w = XtVaCreateManagedWidget("Load",
			      xmPushButtonGadgetClass, action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 1,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNrightAttachment, XmATTACH_POSITION,
			      XmNrightPosition, 1,
			      NULL);
  XtAddCallback(btn_load_w, XmNactivateCallback, 
		(XtCallbackProc) run_load_cb, xmpi_run_w);

  /*
   * Create the "Save" button.
   */
  btn_save_w = XtVaCreateManagedWidget("Save",
			      xmPushButtonGadgetClass, action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 0,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 1,
			      XmNleftAttachment, XmATTACH_POSITION,
			      XmNleftPosition, 1,
			      XmNrightAttachment, XmATTACH_FORM,
			      NULL);
  XtAddCallback(btn_save_w, XmNactivateCallback, 
		(XtCallbackProc) run_save_cb, xmpi_run_w);

  /*
   * Create the "Run" button.
   */
  btn_run_w= XtVaCreateManagedWidget("Run",
			      xmPushButtonGadgetClass, action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 1,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 2,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNrightAttachment, XmATTACH_POSITION,
			      XmNrightPosition, 1,
			      NULL);
  XtAddCallback(btn_run_w, XmNactivateCallback, 
		(XtCallbackProc) run_run_cb, xmpi_run_w);

  /*
   * Create the "Cancel" button.
   */
  btn_cancel_w = XtVaCreateManagedWidget("Cancel",
			      xmPushButtonGadgetClass, action_mgr_w,
			      XmNdefaultButtonShadowThickness, 1,
			      XmNtopAttachment, XmATTACH_POSITION,
			      XmNtopPosition, 1,
			      XmNbottomAttachment, XmATTACH_POSITION,
			      XmNbottomPosition, 2,
			      XmNleftAttachment,XmATTACH_POSITION,
			      XmNleftPosition, 1,
			      XmNrightAttachment,XmATTACH_FORM,
			      NULL);
  XtAddCallback(btn_cancel_w, XmNactivateCallback, 
		(XtCallbackProc) run_cancel_cb, xmpi_run_w);

  XtManageChild(list_w);

  xmpi_nosash(mgr_w);
  XtManageChild(mgr_w);

  /*
   * Build node browser.
   */
  xmpi_nodes_build(dialog_mgr_w);

  // build the vendor window
  if (numb_horz_panes == 3)
    xmpi_vendor_build(dialog_mgr_w);

  XtManageChild(dialog_mgr_w);
  XtManageChild(xmpi_run_wrap_w);
  XtManageChild(xmpi_run_w);

  // the schema will not initially be valid.  Disable btns
  run_disable_btns();

  // Argh.  I hate lesstif.  It takes way too much space to draw this
  // screen on a Lesstif setup.  Force to width of screen to make life
  // doable.
#ifdef LESSTIF_VERSION
  int win_width = 0;
  int screen_width = 0;
  XtVaGetValues(xmpi_run_w, 
		XmNwidth, &win_width,
		NULL);
  screen_width = WidthOfScreen(XtScreen(xmpi_run_w));
  screen_width -= 20;

  if (screen_width < win_width)
    XtVaSetValues(xmpi_run_w,
		  XmNwidth, screen_width,
		  NULL);

#endif // LESSTIF_VERSION

  /*
   * Pop up the window.
   */
  XtPopup(xmpi_run_w, XtGrabNone);
}


/*
 *	xmpi_run
 *
 *	Function:	- runs the supplied application schema
 *	Accepts:	- application schema
 *			- popup widget invoking the run if any
 *	Returns:	- 0 or LAMERROR
 */
int
xmpi_run(char* aschema, Widget w)
{
/*
 * Clean away existing application and trace windows.
 */
  xmpi_ctl_setsensitive(XMPI_BKIVIAT, False);
  if (!xmpi_do_rerun) {
    xmpi_mat_destroy();
  }
  xmpi_kv_popdown();
  xmpi_tr_destroy();
  xmpi_vw_dbsetmode(0);
  xmpi_ctl_clean();
  xmpi_busy();

  if (w) {
    XtPopdown(w);
  }

  /*
   * Run the application.
   */
  xmpi_ctl_setinfo((char*) "Starting application...");
  xmpi_flush();

  if (xmpi_sys_run(aschema, &xmpi_app_procs, &xmpi_app_nprocs)) {
    xmpi_do_rerun = 0;
    xmpi_unbusy();
    xmpi_ctl_resetinfo();
    return (LAMERROR);
  }

  /*
   * Initialize and update the view window.
   */
  if (xmpi_do_rerun && xmpi_vw_getprocs()) {
    xmpi_vw_update(1);
  } else {
    xmpi_vw_init();
  }

  xmpi_do_rerun = 0;

  xmpi_ctl_setsensitive(XMPI_BTRACE, True);
  xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, True);
  xmpi_ctl_setsensitive(XMPI_BRERUN, True);
  xmpi_ctl_setsensitive(XMPI_BMATRIX, True);
  xmpi_ctl_resetinfo();
  xmpi_unbusy();

  return (0);
}


/*
 *	xmpi_run_set_prog
 *
 *	Function:	- sets the prog text field
 *			- has side effect of changing base directory
 *	Accepts:	- program filename
 *			- program directory
 */
void
xmpi_run_set_prog(char* prog, char* dir)
{
  strcpy(prog_dir, dir);

  (void) chdir(prog_dir);

  if (prog && *prog) {
    XmTextFieldSetString(prog_w, prog);
  }
}

/*
 *	xmpi_run_delete
 *
 *	Function:	- deletes all selected programs
 */
void
xmpi_run_delete()
{
  int *pposn;			       /* item position array */

  int nposn;			       /* # positions */

  while (XmListGetSelectedPos(list_w, &pposn, &nposn)) {
    XmListDeletePos(list_w, pposn[0]);
    XtFree((char *) pposn);
  }
}

/*
 *	xmpi_run_rerun
 *
 *	Function:	- reruns previous application schema
 */
void
xmpi_run_rerun()
{
  if (custom) {
    xmpi_do_rerun = 1;
    xmpi_run(custom, (Widget) 0);
  }
}

/*
 *	xmpi_run_cleanup
 *
 *	Function:	- removes custom app schema file, if any
 *                      - tells the vendor to clean up...
 */
void
xmpi_run_cleanup()
{
  if (custom) {
    unlink(custom);
    custom = 0;
  }
}

/*
 *	run_run_cb
 *
 *	Function:	- creates application schema from appl list
 *			- runs the application
 *	Accepts:	- parent widget
 */
static void
run_run_cb(Widget w)
{
  xmpi_run_cleanup();

  strcpy(tempfile, "/tmp/xmpiXXXXXX");
  mkstemp(tempfile);
  custom = tempfile;

  if (!run_save(custom)) {
    XtUnmapWidget(xmpi_wmshell(w));

    if (!xmpi_run(custom, xmpi_run_w)) {
      xmpi_ctl_setapp((char*) "<custom built>");
    }
  }
}

/*
 *	run_save
 *
 *	Function:	- saves appl list contents to a file
 *	Accepts:	- save filename
 *	Returns:	- 0 or LAMERROR
 */
static int
run_save(char *filename)
{
  int fd;
  int i;
  int nitems;			       /* # appl list items */
  XmString *items;		       /* appl list items */
  char *progline;		       /* one line in appl schema */
  char* header;
  XtVaGetValues(list_w,
		XmNitemCount, &nitems,
		XmNitems, &items,
		NULL);

  if (nitems <= 0) {
    xmpi_error(xmpi_run_w, (char*) "No programs specified.", false);

    return (LAMERROR);
  }
  fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);

  if (fd < 0) {
    xmpi_error(xmpi_run_w, filename);
    return (LAMERROR);
  }

  if (xmpi_sys_get_run_options(run_options, run_options_length, &header)) {
    xmpi_error(xmpi_run_w, (char*) "Error getting vendor options", false);
    return LAMERROR;
  }

  write(fd, header, strlen(header));
  write(fd, "\n", 1);

  for (i = 0; i < nitems; ++i) {
    XmStringGetLtoR(items[i], XmSTRING_DEFAULT_CHARSET,
		    &progline);
    write(fd, progline, strlen(progline));
    write(fd, "\n", 1);
    XtFree(progline);
  }

  free(header);
  close(fd);
  return (0);
}


// run_change_cb
//
// DESCRIPTION: update schema list on a change
//
// INPUT:  W the client Widget
//         client_data
//         call_cata
//
static void
run_change_cb(Widget, XtPointer, XtPointer)
{
  xmpi_run_update();
}


/*
 *	run_update
 *
 *	Function:	- update a line in the application schema
 *                      - grey / ungrey buttons...
 *
 *      NOTE: THIS FUNCTION ONLY RUNS IF UPDATE IS TRUE!!!
 */
void
xmpi_run_update()
{
  if (!update)
    return;

  int result;
  XmString xstr;
  char *args;
  char *copies;
  char *prog;
  char *linebuf;
  int ncopy = 0;
  char *full_prog;
  bool valid = true;
    
    // Retrieve program name.
  prog = XmTextFieldGetString(prog_w);

  if ((prog == 0) || (*prog == '\0')) {
    run_status_change((char*) "Program Name Not Specified");
    valid = false;
  }

  // Retrieve target copies. Add extra room for -c.
  copies = xmpi_copies_get();

  if ((copies != 0) && (*copies != '\0')) {
    ncopy = stoi((unsigned char *) copies);
  }

  // Retrieve arguments.
  args = XmTextFieldGetString(args_w);

  // need to leave space for the possible / in path.
  full_prog = (char*) malloc(sizeof(char) * (strlen(prog_dir) + 
					     strlen(prog) + 
					     + 2));
  if (full_prog == NULL) {
    xmpi_error(xmpi_run_w, (char*) "malloc()");
    return;
  }

  // Retrieve fullpath toggle.
  if (XmToggleButtonGetState(fullpath_w)) {
    strcpy(full_prog, "");
    if (prog[0] != STRDIR && prog_dir[0]) {
      strcpy(full_prog, prog_dir);
      if (prog_dir[strlen(prog_dir) - 1] != STRDIR) {
	strcat(full_prog, STRSDIR);
      }
    }
    strcat(full_prog, prog);
  } else
    strcpy(full_prog, prog);

  // call the libxmpi side of things...
  result = xmpi_sys_get_schema_line(full_prog, ncopy,  args, 
				    nodes, nodes_length, 
				    schema_options, schema_options_length,
				    &linebuf);
  if (result != 0) {
    char* error_code = NULL;
    error_code = (char*) malloc(sizeof(char) * 
				(strlen("Error in vendor library: ") + 
				 sizeof(int) * 8 + 1));
    if (error_code == NULL) {
      xmpi_error(xmpi_run_w, (char*) "malloc()");
      return;
    }

    xmpi_error(xmpi_run_w, error_code);
    return;
  }

  // Add full program line to application list.
  int total_selected;
  int *selected_pos;
  int selected_count;
  int total_count;
  xstr = XmStringCreateSimple(linebuf);
  XtVaGetValues(list_w,
		XmNitemCount, &total_count,
		XmNselectedItemCount, &total_selected,
		NULL);

  if ((total_count <= 0) || (total_selected <= 0)) {
    XmListAddItem(list_w, xstr, 0);
    XmListSelectPos(list_w, total_count + 1, False);
  } else {
    if (!(XmListGetSelectedPos(list_w, &selected_pos, &selected_count))) {
      xmpi_error(xmpi_run_w, (char*) "No schema line selected.", false);
      return;
    }
    // just replace the first one ;-)
    XmListReplacePositions(list_w, selected_pos, &xstr, 1);
    XmListSelectPos(list_w, selected_pos[0], False);
  }
  XmStringFree(xstr);  
 
  char* message = NULL;
  result = xmpi_sys_verify_schema_line(linebuf, &message);
  if (result != 0) {
    if ((message != NULL) && (valid)) {
      run_status_change(message);
      free(message);
    }
    valid = false;
  }

  // Free things 
  free(linebuf);
  XtFree(prog);
  XtFree(args);

  if (valid) {
    run_enable_btns();
    run_status_change((char*)"");
  } else
    run_disable_btns();
}


/*
 *	run_cancel_cb
 *
 *	Function:	- cancels the dialog with no processing
 */
static void
run_cancel_cb()
{
  XtPopdown(xmpi_run_w);
}


/*
 *	run_save_cb
 *
 *	Function:	- pops up prompt dialog for appl schema filename
 *	Accepts:	- called back from which widget
 */
static void
run_save_cb(Widget from_w)
{
  Widget w;

  XmString str1, str2;

  Arg args[2];

  char *cwd;

  char cwdir[XMPI_PATHMAX];

  cwd = getworkdir();
  if (cwd) {
    strcpy(cwdir, cwd);
    strcat(cwdir, "/");
    free(cwd);
  } else {
    cwdir[0] = 0;
  }

  str1 = XmStringCreateSimple((char*) "Application Schema filename:");
  str2 = XmStringCreateSimple(cwdir);

  XtSetArg(args[0], XmNselectionLabelString, str1);
  XtSetArg(args[1], XmNtextString, str2);

  w = XmCreatePromptDialog(xmpi_wmshell(from_w), 
			   (char*) "save_dialog", args, 2);

  XmStringFree(str1);
  XmStringFree(str2);

  XtAddCallback(w, XmNcancelCallback,
		(XtCallbackProc) XtDestroyWidget, NULL);
  XtAddCallback(w, XmNokCallback, 
		(XtCallbackProc) run_save_ok_cb, NULL);
  XtUnmanageChild(XmSelectionBoxGetChild(w, XmDIALOG_HELP_BUTTON));

  XtManageChild(w);
  XtPopup(XtParent(w), XtGrabNone);
}


/*
 *	run_save_ok_cb
 *
 *	Function:	- extracts selected filename
 *			- saves current appl schema into filename
 *	Accepts:	- selection dialog callback parameters
 */
static void
run_save_ok_cb(Widget parent_w, XtPointer, 
	       XmSelectionBoxCallbackStruct *cbs)
{
  struct stat finfo;		       /* file information */
  char *filename;		       /* appl schema filename */
  int exists = 1;

  XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &filename);

  if (!filename || (*filename == '\0'))
    return;
/*
 * Check if file already exists and if so ask user if they want to overwrite
 * it or not.
 */
  if (stat(filename, &finfo)) {
    if (errno != ENOENT) {
      xmpi_error(parent_w, filename);
      XtFree(filename);
      return;
    }
    exists = 0;
  }
  if (exists && !xmpi_yesno(parent_w, (char*) "File exists. Overwrite?")) {
    XtFree(filename);
    return;
  }
  (void) run_save(filename);
  XtFree(filename);
  XtDestroyWidget(parent_w);
}


/*
 *	xmpi_run_busy
 *
 *	Function:	- set the busy cursor for the Build&Run window
 */
void
xmpi_run_busy()
{
  if (xmpi_run_w)
    xmpi_busy_widget(xmpi_run_w);
}


/*
 *	xmpi_run_unbusy
 *
 *	Function:	- reset the busy cursor for the Build&Run window
 */
void
xmpi_run_unbusy()
{
  if (xmpi_run_w)
    xmpi_unbusy_widget(xmpi_run_w);
}


/*
 *	xmpi_run_file
 *
 *	Function:	- selects program file 
 */
void
xmpi_run_file()
{
  Arg args[2];

  XmString oklabel;

  XmString title;

  Widget help_btn;

  if (fdialog == 0) {
    title = XmStringCreateSimple((char*) "Program Selection");
    oklabel = XmStringCreateSimple((char*) "Ok");

    XtSetArg(args[0], XmNdialogTitle, title);
    XtSetArg(args[1], XmNokLabelString, oklabel);
    fdialog = XmCreateFileSelectionDialog(xmpi_run_w,
					  (char*) "browse_selector", args, 2);

    XtAddCallback(fdialog, XmNokCallback, 
		  (XtCallbackProc) browse_prog_cb, NULL);
    XtAddCallback(fdialog, XmNcancelCallback,
		  (XtCallbackProc) XtUnmanageChild, NULL);

    help_btn = XmFileSelectionBoxGetChild(fdialog,
					  XmDIALOG_HELP_BUTTON);
    XtUnmanageChild(help_btn);

    XmStringFree(title);
    XmStringFree(oklabel);
  } else {
    XmFileSelectionDoSearch(fdialog, 0);
  }

  XtManageChild(fdialog);
}


// browse_prog_cb
// 
// FUNCTION: callback for the build window.  Sets the file name
//           in the program line of the build window.
static void
browse_prog_cb(Widget parent_w, XtPointer, 
	       XmFileSelectionBoxCallbackStruct *cbs)
{
  char *name;			       /* file name */
  char *dir;

  XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &name);
  XmStringGetLtoR(cbs->dir, XmSTRING_DEFAULT_CHARSET, &dir);

  xmpi_run_set_prog(name + strlen(dir), dir);

  XtUnmanageChild(parent_w);
}


static void
run_load_cb(Widget, XtPointer, XtPointer)
{
  Arg args[2];
  XmString oklabel;
  XmString title;
  Widget help_btn;

  if (load_fdialog == 0) {
    title = XmStringCreateSimple((char*) "Schema Selection");
    oklabel = XmStringCreateSimple((char*) "Ok");

    XtSetArg(args[0], XmNdialogTitle, title);
    XtSetArg(args[1], XmNokLabelString, oklabel);
    load_fdialog = XmCreateFileSelectionDialog(xmpi_run_w,
					       (char*) "load_selector", args, 2);

    XtAddCallback(load_fdialog, XmNokCallback, 
		  (XtCallbackProc) load_prog_cb, NULL);
    XtAddCallback(load_fdialog, XmNcancelCallback,
		  (XtCallbackProc) XtUnmanageChild, NULL);

    help_btn = XmFileSelectionBoxGetChild(load_fdialog,
					  XmDIALOG_HELP_BUTTON);
    XtUnmanageChild(help_btn);

    XmStringFree(title);
    XmStringFree(oklabel);
  } else {
    XmFileSelectionDoSearch(load_fdialog, 0);
  }

  XtManageChild(load_fdialog);
}


static void
crop(string& parse_string) 
{
  std::string::size_type start = parse_string.find_first_not_of(" \t\n");
  std::string::size_type end = 
    parse_string.find_last_not_of(" \t\n") - start + 1;

  if (start < end) {
    std::string tmp = parse_string.substr(start, end);
    parse_string = tmp;
  }
}

static void
load_prog_cb(Widget parent_w, XtPointer cdata, 
	       XmFileSelectionBoxCallbackStruct *cbs)
{
  char *name;			       /* file name */
  string line;
  char* message;
  bool cleared = false;
  bool run_options_found = false;
  XmString xstr;

  XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &name);

  fstream input_file;
  input_file.open(name, ios::in);
  if (!input_file) {
    xmpi_error(xmpi_run_w, (char*) "Error opening schema file", false);
    return;
  }

  while (XMPI_GETLINE(input_file, line, '\n')) {
    if (line.find("# mpirun") != std::string::npos)
      if(xmpi_sys_parse_run_options(&run_options, &run_options_length,
				    (char*) line.c_str()) != 0) {
	xmpi_error(xmpi_run_w, (char*) "Error opening schema file", false);
	XmListDeleteAllItems(list_w);
	return;
      } else
	run_options_found = true;

    crop(line);
    if (line[0] != '#' && line.size() != 0) {
      if (xmpi_sys_verify_schema_line((char*) line.c_str(), &message) != 0) {
	free(message);
	xmpi_error(xmpi_run_w, (char*) "Error opening schema file", false);
	XmListDeleteAllItems(list_w);
	return;
      } else {
	xstr = XmStringCreateSimple((char*) line.c_str());
	if (!cleared) {
	  XmListDeleteAllItems(list_w);
	  cleared = true;
	}
	XmListAddItemUnselected(list_w, xstr, 0);
	XmStringFree(xstr);
      }
    }
  }

  if (run_options_found) {
    xmpi_vendor_update_run();
  }

  XmListSelectPos(list_w, 0, True);
  XtUnmanageChild(parent_w);

  free(name);
}


// run_new_cb
//
// FUNCTION: add an <empty> entry to the list if one does not
//           already exist.
//
void
run_new_cb(Widget, XtPointer, XtPointer) 
{
  XmString xstr;
  xstr = XmStringCreateSimple((char*) "<EMPTY>");

  if (! XmListItemExists(list_w, xstr) ) {
    XmListAddItem(list_w, xstr, 0);
    XmListDeselectAllItems(list_w);
    XmListSelectItem(list_w, xstr, True);
  }

  XmStringFree(xstr);
}


static void
run_copy_cb(Widget, XtPointer, XtPointer) 
{
  int selected_count;
  XmString *selected_string;

  XtVaGetValues(list_w, 
		XmNselectedItemCount, &selected_count,
		XmNselectedItems, &selected_string,
		NULL);

  if (selected_count <= 0) {
    xmpi_error(xmpi_run_w, (char*) "No schema line selected.", false);
    return;
  }

  XmListDeselectAllItems(list_w);
  XmListAddItemUnselected(list_w, selected_string[0], 0);
  XmListSelectPos(list_w, 0, True);
}


static void
run_delete_cb(Widget, XtPointer, XtPointer) 
{
  int selected_count;
  int *selected_positions;
  int total_count;

  if (! (XmListGetSelectedPos(list_w, &selected_positions, &selected_count))) {
    xmpi_error(xmpi_run_w, (char*) "No schema line selected.", false);
    return;
  }

  XmListDeletePositions(list_w, selected_positions, selected_count);

  XtVaGetValues(list_w,
		XmNitemCount, &total_count,
		NULL);

  if (total_count > 0)
    if (selected_positions[selected_count - 1] < total_count)
      XmListSelectPos(list_w, selected_positions[selected_count - 1], True);
    else
      XmListSelectPos(list_w, total_count, True);

  XtVaGetValues(list_w,
		XmNitemCount, &total_count,
		NULL);

  if (total_count <= 0) {
    // Add the <EMPTY> line, and select.
    XmString xstr = XmStringCreateSimple((char*) "<EMPTY>");
    XmListAddItem(list_w, xstr, 0);
    XmListSelectItem(list_w, xstr, True);
    XmStringFree(xstr);
  }
}


void
run_select_cb(Widget, XtPointer, XmListCallbackStruct* cb_struct) 
{
  char* prog = NULL;
  int copy = 0;     
  char* args = NULL;
  int result = 0;

  int selected_count;
  int *selected_positions;
  if (! (XmListGetSelectedPos(list_w, &selected_positions, &selected_count))) {
    XmListSelectPos(list_w, cb_struct->item_position, False);
    return;
  }

  XmString *items;
  int items_count;
  char *linebuf;

  XtVaGetValues(list_w,
		XmNitemCount, &items_count,
		XmNitems, &items,
		NULL);

  XmStringGetLtoR(items[(selected_positions[0]) - 1], XmSTRING_DEFAULT_CHARSET,
		  &linebuf);
  
  if (strcmp(linebuf, "<EMPTY>")) {
    xmpi_vendor_clean_options(schema_options, schema_options_length, false);
    
    result = xmpi_sys_parse_schema_line(&prog, &copy, &args,
					&nodes, &nodes_length,
					&schema_options, schema_options_length,
					linebuf);

    if (result != 0) {
      run_status_change((char*) "error parsing schema line");
      return;
    }

    if (prog == NULL)
      prog = strdup("");
    if (args == NULL)
      args = strdup("");
    update = false;
    run_update(prog, copy, args);
    xmpi_nodes_update();
    xmpi_vendor_update_schema();
    update = true;
    XmListSelectPos(list_w, selected_positions[0], False);
  } else {
    prog = strdup("");
    args = strdup("");
    copy = 0;
    update = false;
    run_update(prog, copy, args);
    xmpi_vendor_update_schema_default();
    xmpi_nodes_update_default();
    update = true;
    XmListSelectPos(list_w, selected_positions[0], False);
  }


  XtFree(linebuf);
  XtFree((char*) selected_positions);
  free(prog);
  free(args);
}


static void 
run_enable_btns()
{
  XtVaSetValues(btn_new_w, XmNsensitive, True, NULL);
  XtVaSetValues(btn_copy_w, XmNsensitive, True, NULL);
  XtVaSetValues(btn_save_w, XmNsensitive, True, NULL);
  XtVaSetValues(btn_run_w, XmNsensitive, True, NULL);
}  


static void 
run_disable_btns()
{
  XtVaSetValues(btn_new_w, XmNsensitive, False, NULL);
  XtVaSetValues(btn_copy_w, XmNsensitive, False, NULL);
  XtVaSetValues(btn_save_w, XmNsensitive, False, NULL);
  XtVaSetValues(btn_run_w, XmNsensitive, False, NULL);
}


static void
run_status_change(char* message)
{
  XmString xstr;

  xstr = XmStringCreateSimple(message);
  XtVaSetValues(status_w, XmNlabelString, xstr, NULL);
  XmStringFree(xstr);
}


static void
run_update(char* prog, int copy, char* args)
{
  XtVaSetValues(prog_w, XmNvalue, prog, NULL);
  XtVaSetValues(args_w, XmNvalue, args, NULL);
  xmpi_copies_set(copy);
}
